<?php
/***********************************************************************************
 * The contents of this file are subject to the Extension License Agreement
 * ("Agreement") which can be viewed at
 * https://www.espocrm.com/extension-license-agreement/.
 * By copying, installing downloading, or using this file, You have unconditionally
 * agreed to the terms and conditions of the Agreement, and You may not use this
 * file except in compliance with the Agreement. Under the terms of the Agreement,
 * You shall not license, sublicense, sell, resell, rent, lease, lend, distribute,
 * redistribute, market, publish, commercialize, or otherwise transfer rights or
 * usage to the software or any modified version or derivative work of the software
 * created by or for you.
 *
 * Copyright (C) 2015-2025 Letrium Ltd.
 *
 * License ID: a67be8f75bc1974a987315ddc1024a78
 ************************************************************************************/

namespace Espo\Modules\Sales\Tools\Invoice\Api;

use Espo\Core\Acl;
use Espo\Core\Api\Action;
use Espo\Core\Api\Request;
use Espo\Core\Api\Response;
use Espo\Core\Api\ResponseComposer;
use Espo\Core\Exceptions\BadRequest;
use Espo\Core\Exceptions\Error;
use Espo\Core\Exceptions\Forbidden;
use Espo\Core\Exceptions\NotFound;
use Espo\Modules\Sales\Entities\Invoice;
use Espo\Modules\Sales\Tools\Invoice\EInvoice\Exceptions\RuleValidationFailure;
use Espo\Modules\Sales\Tools\Invoice\EInvoice\Exceptions\UnknownFormat;
use Espo\Modules\Sales\Tools\Invoice\EInvoice\UblGenerator;
use Espo\ORM\EntityManager;

/**
 * @noinspection PhpUnused
 */
class PostValidateEInvoice implements Action
{
    public function __construct(
        private Acl $acl,
        private EntityManager $entityManager,
        private UblGenerator $ublGenerator
    ) {}

    public function process(Request $request): Response
    {
        $id = $request->getRouteParam('id');
        $format = $request->getParsedBody()->format ?? null;

        if (!$this->acl->checkScope(Invoice::ENTITY_TYPE)) {
            throw new Forbidden();
        }

        if (!$id) {
            throw new BadRequest();
        }

        if (!is_string($format)) {
            throw new BadRequest("No format.");
        }

        $entity = $this->getEntity($id);

        try {
            $this->ublGenerator->validate($entity, $format);
        }
        catch (RuleValidationFailure $e) {
            throw Error::createWithBody(
                $e->getMessage(),
                Error\Body::create()
                    ->withMessageTranslation('ublRuleValidationFailure', Invoice::ENTITY_TYPE, [
                        'ruleId' => $e->getRuleId(),
                        'message' => $e->getMessage(),
                    ])
            );
        }
        catch (UnknownFormat) {
            throw new BadRequest("Unknown format.");
        }

        return ResponseComposer::json(true);
    }

    /**
     * @throws Forbidden
     * @throws NotFound
     */
    private function getEntity(string $id): Invoice
    {
        $entity = $this->entityManager->getRDBRepositoryByClass(Invoice::class)->getById($id);

        if (!$entity) {
            throw new NotFound();
        }

        if (!$this->acl->checkEntityRead($entity)) {
            throw new Forbidden();
        }

        return $entity;
    }
}
