<?php
/***********************************************************************************
 * The contents of this file are subject to the Extension License Agreement
 * ("Agreement") which can be viewed at
 * https://www.espocrm.com/extension-license-agreement/.
 * By copying, installing downloading, or using this file, You have unconditionally
 * agreed to the terms and conditions of the Agreement, and You may not use this
 * file except in compliance with the Agreement. Under the terms of the Agreement,
 * You shall not license, sublicense, sell, resell, rent, lease, lend, distribute,
 * redistribute, market, publish, commercialize, or otherwise transfer rights or
 * usage to the software or any modified version or derivative work of the software
 * created by or for you.
 *
 * Copyright (C) 2024-2025 Letrium Ltd.
 *
 * License ID: f27e70ce6801a13265271f5669c8bc5c
 ************************************************************************************/

namespace Espo\Modules\Project\Classes\Acl\Project;

use Espo\Core\Acl\AccessEntityCREDSChecker;
use Espo\Core\Acl\DefaultAccessChecker;
use Espo\Core\Acl\ScopeData;
use Espo\Entities\User;
use Espo\Modules\Project\Entities\Project;
use Espo\ORM\Entity;
use Espo\ORM\EntityManager;

/**
 * @implements AccessEntityCREDSChecker<Project>
 */
class AccessChecker implements AccessEntityCREDSChecker
{
    public function __construct(
        private DefaultAccessChecker $defaultAccessChecker,
        private EntityManager $entityManager,
    ) {}

    public function check(User $user, ScopeData $data): bool
    {
        return $this->defaultAccessChecker->check($user, $data);
    }

    public function checkCreate(User $user, ScopeData $data): bool
    {
        return $this->defaultAccessChecker->checkCreate($user, $data);
    }

    public function checkRead(User $user, ScopeData $data): bool
    {
        if ($data->isFalse()) {
            return false;
        }

        return true;
    }

    public function checkEdit(User $user, ScopeData $data): bool
    {
        if ($data->isFalse()) {
            return false;
        }

        return true;
    }

    public function checkDelete(User $user, ScopeData $data): bool
    {
        return $this->defaultAccessChecker->checkDelete($user, $data);
    }

    public function checkStream(User $user, ScopeData $data): bool
    {
        return $this->checkRead($user, $data);
    }

    public function checkEntityCreate(User $user, Entity $entity, ScopeData $data): bool
    {
        return $this->defaultAccessChecker->checkEntityCreate($user, $entity, $data);
    }

    public function checkEntityRead(User $user, Entity $entity, ScopeData $data): bool
    {
        if ($this->defaultAccessChecker->checkEntityRead($user, $entity, $data)) {
            return true;
        }

        if ($data->isFalse()) {
            return false;
        }

        return $this->checkMember($user, $entity);
    }

    public function checkEntityEdit(User $user, Entity $entity, ScopeData $data): bool
    {
        if ($this->defaultAccessChecker->checkEntityEdit($user, $entity, $data)) {
            return true;
        }

        return $this->checkMember($user, $entity, [Project::ROLE_OWNER, Project::ROLE_EDITOR]);
    }

    public function checkEntityDelete(User $user, Entity $entity, ScopeData $data): bool
    {
        return $this->defaultAccessChecker->checkEntityDelete($user, $entity, $data);
    }

    public function checkEntityStream(User $user, Entity $entity, ScopeData $data): bool
    {
        return $this->checkEntityRead($user, $entity, $data);
    }

    /**
     * @param string[]|string|null $role
     */
    private function checkMember(User $user, Project $entity, array|string|null $role = null): bool
    {
        $relation = $this->entityManager->getRelation($entity, 'members');

        if (!$role) {
            return $relation->isRelated($user);
        }

        $roles = $role;

        if (is_string($role)) {
            $roles = [$role];
        }

        /** @var string[] $roles */

        return in_array($relation->getColumn($user, 'role'), $roles);
    }
}
