/*
 * This file is part of  the extension: Ebla Multiple Link Pro
 * Copyright (c) Eblasoft Bilişim Ltd.
 *
 * This Software is the property of Eblasoft Bilişim Ltd. and is protected
 * by copyright law - it is NOT Freeware and can be used only in one project
 * under a proprietary license, which is delivered along with this program.
 * If not, see <http://eblasoft.com.tr/eula>.
 *
 * This Software is distributed as is, with LIMITED WARRANTY AND LIABILITY.
 * Any unauthorised use of this Software without a valid license is
 * a violation of the License Agreement.
 *
 * According to the terms of the license you shall not resell, sublicense,
 * rent, lease, distribute or otherwise transfer rights or usage of this
 * Software or its derivatives. You may modify the code of this Software
 * for your own needs, if source code is provided.
 */

define('ebla-link-pro:views/fields/link-multiple-form', 'views/fields/link-multiple', function (Dep) {

    return Dep.extend({

        detailTemplate: 'ebla-link-pro:fields/link-multiple-form/detail',

        listView: 'ebla-link-pro:views/fields/link-multiple-form/inline-list',

        noCreateScopeList: ['User', 'Team', 'Role', 'Portal'],

        validations: ['required'],

        validationElementSelector: '.add-item',

        relatedAttributeFunctions: {},

        relatedAttributeMap: {},

        forceSelectAllAttributes: true,

        listOptions: {},

        noFullForm: false,

        isManyToMany: false,

        validateRequired: function () {
            if (!this.form || this.noAccess) return Dep.prototype.validateRequired.call(this);

            if (this.isRequired()) {
                if (this.collection.length === 0) {
                    const msg = this.translate('fieldIsRequired', 'messages').replace('{field}', this.getLabelText());
                    this.showValidationMessage(msg);
                    return true;
                }
            }
        },

        validate: function () {
            let notValid = Dep.prototype.validate.call(this);

            if (!this.form || this.noAccess || this.isManyToMany) return notValid;

            const listView = this.getView('list');

            const rowViews = listView.nestedViews;
            const ids = this.collectionPluck('id');
            ids.forEach((id) => {
                try {
                    const rowView = rowViews[id];

                    Object.keys(rowView.nestedViews).forEach(function (field) {
                        const fieldView = rowView.nestedViews[field];
                        const fieldName = field.replace('Field', '');

                        if (
                            typeof fieldView.validate === 'function' &&
                            fieldView.isEditMode() &&
                            fieldView.model.hasField(fieldName)
                        ) {
                            fieldView.fetchToModel();
                            notValid = fieldView.validate() || notValid;
                        }
                    });
                } catch (e) {
                    console.log(e);
                }
            });

            return notValid
        },

        events: _.extend({
            'click [data-action="addItem"]': function (e) {
                this.openModal();
            },
            'click [data-action="addRow"]': function (e) {
                this.createRelated();
            },
            'click [data-action="createInModal"]': function (e) {
                this.create();
            },
            'click [data-action="addMultipleItems"]': function (e) {
                this.openModal();
            },
            'click button[data-action="quickCreate"]': function (e) {
                const scope = e.currentTarget.getAttribute('data-value');
                this.quickCreate(scope);
            }
        }, Dep.prototype.events),

        data: function () {
            const data = Dep.prototype.data.call(this);
            data.form = this.form;
            data.select = this.enableSelect;
            data.enableAdd = this.add && this.enableAdd;
            data.enableCreateInModal = this.enableCreateInModal;
            data.enableAddMultiple = this.enableAddMultiple;
            data.hasNumericFields = this.hasNumericFields;
            data.middleLink = this.middleLink;
            data.add = this.add;
            data.createForeignLabel = this.translate('Create ' + this.foreignScope, 'labels', this.foreignScope);
            data.top = this.buttonsPosition === "top";

            data.middleLinkIsParent = Object.keys(this.parentSelectButtons || {}).length > 0;
            data.parentSelectButtons = this.parentSelectButtons || {};

            return data;
        },

        init: function () {
            Dep.prototype.init.call(this);

            this.defs = this.model.defs.fields[this.name];

            this.readOnlyFields = this.defs['readOnlyFields'];
            this.form = this.defs['enableForm'] && !['list', 'search'].includes(this.mode);
            this.layoutName = this.defs['layout'] || 'listSmall';
            this.editLayoutName = this.defs['editLayout'] === 'sameAsViewMode' ? this.layoutName : this.defs['editLayout'] || this.layoutName;
            this.enableSelect = this.defs['enableSelect'];
            this.enableAdd = this.defs['enableAdd'];
            this.enableCreateInModal = this.defs['enableCreateInModal'];
            this.enableAddMultiple = this.defs['enableAddMultiple'];
            this.footerSumFields = this.defs['footerSumFields'];
            this.footerSumFieldsMap = this.defs['footerSumFieldsMap'];
            this.middleLink = this.defs['middleLink'];
            this.isMiddleLinkUnique = this.defs['isMiddleLinkUnique'];
            this.defaultOrderBy = this.defs['orderBy'];
            this.defaultOrder = this.defs['order'];
            this.buttonsPosition = this.defs['buttonsPosition'];
        },

        _getTemplateName: function () {
            if (
                this.isEditMode() &&
                !this.isManyToMany &&
                this.form
            ) return 'ebla-link-pro:fields/link-multiple-form/edit';

            return Dep.prototype._getTemplateName.call(this);
        },

        createRelated: function (data, n = 1) {
            data = data || {};

            const mId = this.middleLink + 'Id';
            if (this.middleLink && this.isMiddleLinkUnique) {
                if (this.collectionPluck(mId).includes(data[mId])) {
                    return;
                }
            }

            this.getModelFactory().create(this.scope, function (entity) {
                entity.populateDefaults();
                entity.set(data);

                for (let i = 0; i < n; i++) {
                    const model = entity.clone();
                    model.set('id', data.id || this.generateId());
                    model.isNewItem = true;

                    if (!this.model.isNew()) {
                        model.setRelate({
                            model: this.model,
                            link: this.foreignLink
                        });
                    }
                    this.collection.add(model);
                }

                setTimeout(() => {
                    this.collection.trigger('sync');
                    this.trigger('change'); // this will fetch the value to model
                }, 150);
            }.bind(this));
        },

        linkModels: function (models) {
            models.forEach(function (model) {
                if (this.middleLink) {
                    const attrs = {};
                    attrs[this.middleLink + 'Id'] = model.id;
                    attrs[this.middleLink + 'Name'] = model.get('name');

                    const copyFields = this.model.getFieldParam(this.name, 'copyFields') || [];
                    copyFields.forEach((field) => {
                        this.getFieldManager().getEntityTypeFieldAttributeList(model.name, field).forEach(function (attribute) {
                            attrs[attribute] = model.get(attribute);
                        }, this);
                    });

                    this.createRelated(this.mapAttrs(attrs, model));
                } else {
                    if (!this.model.isNew()) {
                        model.setRelate({
                            model: this.model,
                            link: this.foreignLink
                        });
                    }
                    model.isNewItem = true;
                    this.collection.add(model);
                }
            }.bind(this));

            this.collection.trigger('sync');
        },

        mapAttrs: function (attrs, model) {
            return attrs;
        },

        generateId: function () {
            return 'cid' + Math.random().toString(36).substr(2, 10);
        },

        openModal: function () {
            let attributes = {};
            const n = this.$el.find('input.no-of-items').val() || 1;

            if (this.relatedAttributeFunctions[this.link] && typeof this.relatedAttributeFunctions[this.link] === 'function') {
                attributes = _.extend(this.relatedAttributeFunctions[this.link].call(this), attributes);
            }

            Object.keys(this.relatedAttributeMap[this.link] || {}).forEach(function (attr) {
                attributes[this.relatedAttributeMap[this.link][attr]] = this.model.get(attr);
            }, this);

            this.notify('Loading...');

            const viewName = this.getMetadata().get('clientDefs.' + this.foreignScope + '.modalViews.edit') || 'ebla-link-pro:views/modals/add';
            const relate = this.middleLink ? null : {
                model: this.model,
                link: this.foreignLink,
            };
            this.createView('quickCreate', viewName, {
                relate,
                scope: this.foreignScope,
                attributes: attributes,
                dontSave: !this.middleLink
            }, function (view) {
                view.render();
                view.notify(false);
                this.listenToOnce(view, 'found-duplicates', function (duplicates) {
                    this.notify(false);
                    model.set(duplicates[0]);
                    this.linkModels([model]);
                }, this);
                this.listenToOnce(view, 'after:save', function (model) {
                    if (this.middleLink) {
                        Espo.Ui.success(this.getLanguage().translate('Already exist, linked!'));
                        this.linkModels([model]);
                    } else {
                        this.createRelated(model.attributes, n);
                    }
                }, this);
                this.listenToOnce(view, 'add', function (model) {
                    view.notify(false);
                    if (this.middleLink) {
                        Espo.Ui.success(this.getLanguage().translate('Already exist, linked!'));
                        this.linkModels([model]);
                    } else {
                        this.createRelated(model.attributes, n);
                    }
                }, this);
            }.bind(this));
        },

        setFScope: function () {
            if (this.mode === 'edit') {
                this.foreignScope = this.getMetadata().get(['entityDefs', this.foreignScope, 'links', this.middleLink, 'entity']) || this.foreignScope;
            } else {
                this.foreignScope = this._foreignScope;
            }
        },

        setup: function () {
            Dep.prototype.setup.call(this);

            if (!this.form || this.mode === 'search' || this.mode === 'list') return;

            this.panelDefs = this.panelDefs || {};

            if (this.middleLink) {
                const middleLinkType = this.getMetadata().get('entityDefs.' + this.foreignScope + '.fields.' + this.middleLink + '.type');
                this._foreignScope = this.foreignScope + '';
                this.setFScope();
                this.on('mode-changed', () => {
                    this.setFScope();
                });

                if (middleLinkType === 'linkParent') {
                    const parents = this.getMetadata().get(['entityDefs', this.foreignScope, 'fields', this.middleLink, 'entityList']) || [];
                    this.parentSelectButtons = {};
                    parents.forEach((parent) => {
                        this.parentSelectButtons[parent] = this.getMetadata().get(['clientDefs', parent, 'iconClass']);
                    });
                }
            }

            if (this.enableSelect) {
                this.addActionHandler('selectLinkPlus', () => {
                    this.actionSelect(true);
                });
            }

            this.link = this.name;
            this.scope = this.model.defs.links[this.link].entity;
            this.foreignLink = this.model.defs['links'][this.link]['foreign'];
            this.isManyToMany = this.getMetadata().get('entityDefs.' + this.foreignScope + '.links.' + this.foreignLink + '.type') === 'hasMany';

            if (!this.isManyToMany) {
                for (const key in this.events) {
                    if (this.events.hasOwnProperty(key) && key === 'click button[data-action="selectLink"]') {
                        delete this.events[key];
                    }
                }
            }

            if (!this.getAcl().check(this.scope, 'read')) {
                this.noAccess = true;
                return;
            }

            const url = this.url = this.url || [this.model.name, this.model.id, this.link].join('/');

            if (
                this.getAcl().check(this.scope, 'create') &&
                !~this.noCreateScopeList.indexOf(this.scope)
            ) {
                this.add = true;
            }

            this.wait(true);
            this.getCollectionFactory().create(this.scope, function (collection) {
                collection.maxSize = this.getConfig().get('recordListMaxSizeLimit') || 200;
                collection.url = collection.urlRoot = url;
                collection.parentModel = this.model;

                if (this.defaultOrderBy) {
                    collection.orderBy = this.defaultOrderBy;
                    collection.defaultOrderBy = this.defaultOrderBy;
                }
                if (this.defaultOrder) {
                    collection.order = this.defaultOrder;
                    collection.defaultOrder = this.defaultOrder;
                }

                this.collection = collection;

                this.collection.on('model-removing', (id) => {
                    this.model.trigger('change:' + this.name);
                    this.trigger('model-removing', id);
                });

                this.collection.on('change', () => {
                    this.trigger('change');
                });

                this.hasNumericFields = false;

                if (_.isArray(this.footerSumFields) && this.footerSumFields.length > 0) {
                    this.hasNumericFields = true;
                    const viewName = this.getMetadata().get('entityDefs.' + this.model.name + '.fields.' + this.name + '.footerView') || 'ebla-link-pro:views/fields/link-multiple-form/footer-sum';
                    this.createView(
                        'footerSum',
                        viewName, {
                            el: this.options.el + ' .footer-sum',
                            collection: collection,
                            scope: this.foreignScope,
                            numericFields: this.footerSumFields || [],
                            numericFieldsMap: this.footerSumFieldsMap || {},
                        }, null);
                }

                this.wait(false);
            }, this);
        },

        actionSelect(plus = false) {
            if (!this.form || this.isManyToMany || !plus) {
                // to disable select for inline link multiple fields
                return Dep.prototype.actionSelect.call(this);
            }

            Espo.Ui.notify(' ... ');

            const panelDefs = this.panelDefs;

            const viewName = panelDefs.selectModalView ||
                this.getMetadata().get(`clientDefs.${this.foreignScope}.modalViews.select`) ||
                this.selectRecordsView;

            const mandatorySelectAttributeList = this.mandatorySelectAttributeList ||
                panelDefs.selectMandatoryAttributeList;

            const createButton = this.isEditMode() &&
                (!this.createDisabled && !panelDefs.createDisabled || this.forceCreateButton);

            let createAttributesProvider = null;

            if (createButton) {
                createAttributesProvider = () => {
                    let attributes = this.getCreateAttributes() || {};

                    if (!panelDefs.createHandler) {
                        return Promise.resolve(attributes);
                    }

                    return new Promise(resolve => {
                        Espo.loader.requirePromise(panelDefs.createHandler)
                            .then(Handler => new Handler(this.getHelper()))
                            .then(handler => {
                                handler.getAttributes(this.model)
                                    .then(additionalAttributes => {
                                        resolve({
                                            ...attributes,
                                            ...additionalAttributes,
                                        });
                                    });
                            });
                    });
                };
            }

            this._getSelectFilters().then(filters => {
                this.createView('dialog', viewName, {
                    scope: this.foreignScope,
                    createButton: createButton,
                    filters: filters.advanced,
                    boolFilterList: filters.bool,
                    primaryFilterName: filters.primary,
                    filterList: this.getSelectFilterList(),
                    multiple: true,
                    mandatorySelectAttributeList: mandatorySelectAttributeList,
                    forceSelectAllAttributes: this.forceSelectAllAttributes,
                    createAttributesProvider: createAttributesProvider,
                    layoutName: this.panelDefs.selectLayout,
                }, dialog => {
                    dialog.render();

                    Espo.Ui.notify(false);

                    this.listenToOnce(dialog, 'select', models => {
                        this.clearView('dialog');

                        if (Object.prototype.toString.call(models) !== '[object Array]') {
                            models = [models];
                        }

                        this.linkModels(models); // this is the change
                    });
                });
            });
        },

        /* this is copy-paste from core just for backward compatibility */
        _getSelectFilters() {
            const handler = this.panelDefs.selectHandler;

            if (!handler || this.isSearchMode()) {
                return Promise.resolve({});
            }

            return new Promise(resolve => {
                Espo.loader.requirePromise(handler)
                    .then(Handler => new Handler(this.getHelper()))
                    .then(/** module:handlers/select-related */handler => {
                        return handler.getFilters(this.model);
                    })
                    .then(filters => {
                        const advanced = {...(this.getSelectFilters() || {}), ...(filters.advanced || {})};
                        const primaryFilter = this.getSelectPrimaryFilterName() ||
                            filters.primary || this.panelDefs.selectPrimaryFilterName;

                        const localBoolFilterList = this.getSelectBoolFilterList();

                        const boolFilterList = (localBoolFilterList || filters.bool || this.panelDefs.selectBoolFilterList) ?
                            [
                                ...(localBoolFilterList || []),
                                ...(filters.bool || []),
                                ...(this.panelDefs.selectBoolFilterList || []),
                            ] :
                            undefined;

                        resolve({
                            bool: boolFilterList,
                            primary: primaryFilter,
                            advanced: advanced,
                        });
                    });
            });
        },

        getListOptions: function (callback) {
            const options = _.extend({
                collection: this.collection,
                checkboxes: false,
                mode: this.mode,
                model: this.model,
                buttonsDisabled: true,
                el: this.options.el + ' .inline-list',
                skipBuildRows: true,
                rowActionsDisabled: (this.inlineEditDisabled || this.readOnly) && !this.isEditMode(),
                noFullForm: this.noFullForm,
                requiredFields: this.requiredFields,
                readOnlyFields: this.readOnlyFields,
                rowActionsOptions: {
                    unlinkDisabled: this.unlinkDisabled,
                    actions: this.readOnly ? [] : this.model.getFieldParam(this.name, 'rowActionItems'),
                    mode: this.mode,
                }
            }, this.listOptions);

            options.layoutName = (this.mode === 'edit') ? this.editLayoutName : this.layoutName;

            callback(options);
        },

        afterRender: function () {
            Dep.prototype.afterRender.call(this);

            if (this.getConfig().get('unlinkButtonIconClass')) {
                this.$el.find('.fa-times')?.addClass(this.getConfig().get('unlinkButtonIconClass'));
            }

            if (!this.form) {
                return;
            }

            if (this.noAccess) {
                this.hide();
                return;
            }


            if (this.isManyToMany && this.isEditMode()) {
                return;
            }

            this.collection.reset();
            this.collection.set(this.model.get(this.name + 'ListHolder'));
            this.collection.total = this.collection.length;

            const duplicateIgnore = this.model.getFieldParam(this.name, 'duplicateIgnore');
            if (duplicateIgnore !== true) {
                this.duplicateSourceId = this.model.get('_duplicatingEntityId') || this._parentView?._parentView?.options?.duplicateSourceId;
                if (this.duplicateSourceId) {
                    this.collection.models.forEach((model) => {
                        model.isNewItem = true;
                        model.id = this.generateId();
                        model.set('id', model.id);
                    });
                }
            }

            this.getListOptions((options) => {
                const viewName = this.model.getFieldParam(this.name, 'listView') ||
                    this.options.listView ||
                    this.listView;

                this.createView('list', viewName, options, (inlineListView) => {
                    inlineListView.getSelectAttributeList((selectAttributeList) => {
                        if (this.mandatorySelectAttributeList) {
                            selectAttributeList = selectAttributeList.concat(this.mandatorySelectAttributeList);
                        }
                        if (selectAttributeList) {
                            this.collection.data.select = selectAttributeList.join(',');
                        }
                        this.collection.trigger('sync');
                    });

                    inlineListView.on('after:save', (model) => {
                        this.trigger('change');
                        this.trigger('model-updating', model);
                    });

                    const updateRecordViewAttrs = (id) => {
                        const parentView = this.getParentView();
                        const attributes = parentView?.recordViewObject?.attributes || parentView?.getParentView()?.attributes;

                        if (attributes) {
                            attributes[this.name + 'ListHolder'] = _.filter(attributes[this.name + 'ListHolder'], (item) => {
                                return item?.id !== id;
                            });
                        }

                        this.trigger('change');
                    };

                    inlineListView.on('after:unrelate', (_options) => {
                        _options = _options || {};
                        const {inlineList, id} = _options;
                        if (inlineList) {
                            updateRecordViewAttrs(id);
                        }
                    });

                    inlineListView.on('after:inline-remove', (_options) => {
                        _options = _options || {};
                        const {id} = _options;
                        updateRecordViewAttrs(id);
                    });
                });
            });
        },

        fetch: function () {
            const data = Dep.prototype.fetch.call(this);

            if (!this.form || this.noAccess || this.isManyToMany) {
                return data;
            }

            data[this.name + 'ListHolder'] = this.collectionToJSON();

            if (this.isManyToMany) {
                return data;
            }

            // check if espo 8
            if (typeof Espo.loader.addBundleMapping === 'function') {
                data[this.name + 'Ids'] = this.collectionPluck('id');
                data[this.name + 'Names'] = this.collectionPluck('name');
            }

            return data;
        },

        addLink: function (id, name) {
            if (this.isManyToMany) {
                this.getModelFactory().create(this.foreignScope, model => {
                    model.set({
                        id: id,
                        name: name,
                    })

                    this.collection.add(model);
                })
            }

            return Dep.prototype.addLink.call(this, id, name);
        },

        deleteLink: function (id) {
            if (this.isManyToMany) {
                this.collection.remove(id);
            }

            return Dep.prototype.deleteLink.call(this, id)
        },

        create: function () {
            Espo.Ui.notify(' ... ');

            const viewName = this.getMetadata()
                    .get(['clientDefs', this.collection?.name, 'modalViews', 'edit']) ||
                'views/modals/edit';

            const attributes = {};

            const attributeMap = this.getMetadata()
                .get(['clientDefs', this.entityType, 'relationshipPanels', this.link, 'createAttributeMap']) || {};

            Object.keys(attributeMap)
                .forEach(attr => {
                    attributes[attributeMap[attr]] = this.model.get(attr);
                });

            const foreignLink = this.model.defs['links'][this.link].foreign;

            this.createView('quickCreate', viewName, {
                scope: this.collection?.name || this.foreignScope,
                fullFormDisabled: true,
                relate: {
                    model: this.model,
                    link: foreignLink,
                },
                attributes: attributes,
            }, view => {
                view.render()
                    .then(() => Espo.Ui.notify(false));

                this.listenToOnce(view, 'leave', () => {
                    view.close();
                    this.close();
                });

                this.listenToOnce(view, 'after:save', (model) => {
                    view.close();

                    if (!this.model.isNew()) {
                        model.setRelate({
                            model: this.model,
                            link: this.foreignLink
                        });
                    }
                    model.isNewItem = true;
                    this.collection.add(model);
                    this.collection.trigger('sync');
                });
            });
        },

        collectionToJSON: function () {
            const data = [];

            this.collection.forEach((model) => {
                if (model.getClonedAttributes) {
                    data.push(model.getClonedAttributes());
                } else {
                    data.push(model.toJSON());
                }
            });

            return data;
        },

        collectionPluck: function (attr) {
            const data = [];
            this.collection.forEach((model) => {
                data.push(model.get(attr));
            });
            return data;
        },

        quickCreate(scope) {
            const viewName =
                this.getMetadata().get(`clientDefs.${scope}.modalViews.select`) ||
                this.selectRecordsView;

            const mandatorySelectAttributeList = this.mandatorySelectAttributeList;

            this.createView('dialog', viewName, {
                scope: scope,
                createButton: false,
                multiple: false,
                mandatorySelectAttributeList: mandatorySelectAttributeList,
            }, dialog => {
                dialog.render();

                Espo.Ui.notify(false);

                this.listenToOnce(dialog, 'select', model => {
                    this.clearView('dialog');

                    if (Object.prototype.toString.call(model) === '[object Array]') {
                        model = model[0];
                    }

                    const data = {};
                    data[this.middleLink + 'Id'] = model.id;
                    data[this.middleLink + 'Name'] = model.get('name');
                    data[this.middleLink + 'Type'] = scope;

                    this.createRelated(data);
                });
            });
        },
    });
});
