/*
 * This file is part of  the extension: Ebla Multiple Link Pro
 * Copyright (c) Eblasoft Bilişim Ltd.
 *
 * This Software is the property of Eblasoft Bilişim Ltd. and is protected
 * by copyright law - it is NOT Freeware and can be used only in one project
 * under a proprietary license, which is delivered along with this program.
 * If not, see <http://eblasoft.com.tr/eula>.
 *
 * This Software is distributed as is, with LIMITED WARRANTY AND LIABILITY.
 * Any unauthorised use of this Software without a valid license is
 * a violation of the License Agreement.
 *
 * According to the terms of the license you shall not resell, sublicense,
 * rent, lease, distribute or otherwise transfer rights or usage of this
 * Software or its derivatives. You may modify the code of this Software
 * for your own needs, if source code is provided.
 */

(function () {
    function getSource(Dep) {
        const _setup = Dep.prototype.setup;
        const _afterRender = Dep.prototype.afterRender;
        const _select = Dep.prototype.select;

        Dep.prototype.lastSearchKeyword = null;
        Dep.prototype.quickCreateDisabled = true;

        return {
            setup: function () {
                _setup.call(this);

                const createButton = this.createButton && (!this.createDisabled || this.forceCreateButton);

                if (
                    !this.isSearchMode() &&
                    (
                        this.foreignScope &&
                        this.getAcl().check(this.foreignScope, 'create')
                    ) &&
                    !this.getConfig().get('disableEblaLinkProQuickCreate') &&
                    createButton &&
                    !this.autocompleteOnEmpty
                ) {
                    this.quickCreateDisabled = false;
                }
            },

            afterRender: function () {
                _afterRender.call(this);

                this.setAutoElement();

                if (!this.isEditMode() || this.autocompleteDisabled || this.quickCreateDisabled) return;

                // remove + button
                this.$el.find('button[data-action="createLink"]').remove();

                this.$autoElement.autocomplete('setOptions', {
                    minChars: 0,
                    lookup: (query, done) => {
                        this.eblaLookupFunction(this, query).then(items => {
                            this.lastSearchKeyword = this.$autoElement.val().trim();

                            const createItem = {
                                label: this.translate('Create ' + this.foreignScope, 'labels', this.foreignScope) + ' ...',
                                value: this.lastSearchKeyword,
                                id: 'create-new',
                                data: this.lastSearchKeyword,
                                attributes: {
                                    id: 'create-new',
                                }
                            };

                            if (this.lastSearchKeyword) {
                                createItem.label = this.translate('Create') + ' ' + this.lastSearchKeyword + ' ...';
                            }

                            createItem.value = createItem.label;

                            items.push(createItem);

                            done({suggestions: items});
                        });
                    },
                    beforeRender: ($container, suggestions) => {
                        $container.addClass('has-quick-create');
                    }
                });
            },

            setAutoElement: function () {
                this.$autoElement = this.$elementName;
            },

            openModalQuickCreate: function () {
                Espo.Ui.notify(' ... ');

                const viewName = this.getMetadata().get(['clientDefs', this.foreignScope, 'modalViews', 'edit']) || 'views/modals/edit';

                const attributes = this.getCreateAttributes() || {};
                attributes.firstName = this.lastSearchKeyword?.split(' ')[0];
                attributes.lastName = this.lastSearchKeyword?.split(' ')[1];
                attributes.name = this.lastSearchKeyword;

                if (!this.model.isNew()) {
                    const linkField = Espo.Utils.lowerCaseFirst(this.entityType);
                    attributes[linkField + 'Id'] = this.model.id;
                    attributes[linkField + 'Name'] = this.model.get('name');
                }

                this.createView('quickCreate', viewName, {
                    scope: this.foreignScope,
                    fullFormDisabled: true,
                    attributes,
                }, view => {
                    view.render()
                        .then(() => Espo.Ui.notify(false));

                    this.listenToOnce(view, 'leave', () => {
                        view.close();
                    });

                    this.listenToOnce(view, 'after:save', (model) => {
                        view.close();

                        if (typeof this.addLink === 'function') {
                            this.addLink(model.id, model.get('name'));
                        }
                        if (typeof this.select === 'function') {
                            this.select(model);
                        }
                    });
                });
            },

            select: function (model) {
                if (model.id === 'create-new') {
                    return this.openModalQuickCreate();
                }

                return _select.call(this, model);
            },
        };
    }

    Espo.require('views/fields/link', function (Dep) {
        _.extend(Dep.prototype, getSource(Dep), {
            eblaLookupFunction: (context, query) => {
                if (!context.autocompleteOnEmpty && query.length === 0) {
                    isEmptyQueryResult = true;

                    const emptyResult = context.getEmptyAutocompleteResult();

                    if (emptyResult) {
                        return Promise.resolve(context._transformAutocompleteResult({list: emptyResult}));
                    }

                    return Promise.resolve([]);
                }

                isEmptyQueryResult = false;

                return Promise.resolve(context.getAutocompleteUrl(query))
                    .then(url => Espo.Ajax.getRequest(url, {q: query}))
                    .then(response => context._transformAutocompleteResult(response));
            }
        });
    });

    Espo.require('views/fields/link-multiple', function (Dep) {
        const _addLink = Dep.prototype.addLink;

        _.extend(Dep.prototype, getSource(Dep), {
            setAutoElement: function () {
                this.$autoElement = this.$element;
            },

            eblaLookupFunction: (context, query) => {
                if (!query) {
                    return Promise.resolve([]);
                }

                return Promise.resolve(context.getAutocompleteUrl(query))
                    .then(url => Espo.Ajax.getRequest(url, {q: query}))
                    .then(/** {list: Record[]} */response => {
                        return response.list.map(item => ({
                            value: item.name,
                            attributes: item,
                        }));
                    });
            },

            addLink: function (id, name) {
                if (id === 'create-new') {
                    return this.openModalQuickCreate();
                }

                _addLink.call(this, id, name);
            },
        });
    });

    Espo.require('views/fields/link-parent', function (Dep) {
        _.extend(Dep.prototype, getSource(Dep), {
            eblaLookupFunction: (context, query) => {
                if (!query) {
                    return Promise.resolve([]);
                }

                return Promise.resolve(context.getAutocompleteUrl(query))
                    .then(url => Espo.Ajax.getRequest(url, {q: query}))
                    .then(/** {list: Record[]} */response => {
                        return response.list.map(item => ({
                            value: item.name,
                            attributes: item,
                        }));
                    });
            },
        });
    });
}).call()
