<?php
/************************************************************************
 * This file is part of EspoCRM.
 *
 * EspoCRM - Open Source CRM application.
 * Copyright (C) 2014-2021 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
 * Website: https://www.espocrm.com
 *
 * EspoCRM is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * EspoCRM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with EspoCRM. If not, see http://www.gnu.org/licenses/.
 *
 * The interactive user interfaces in modified source and object code versions
 * of this program must display Appropriate Legal Notices, as required under
 * Section 5 of the GNU General Public License version 3.
 *
 * In accordance with Section 7(b) of the GNU General Public License version 3,
 * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
 ************************************************************************/

namespace Espo\Core\FieldProcessing\Stream;

use Espo\ORM\Entity;

use Espo\Core\{
    FieldProcessing\Loader as LoaderInterface,
    FieldProcessing\Loader\Params,
    Utils\Metadata,
    Utils\Config,
    Acl,
};

use Espo\Entities\User;

use Espo\Services\Stream as StreamService;

class FollowersLoader implements LoaderInterface
{
    private const FOLLOWERS_LIMIT = 5;

    private $streamService;

    private $metadata;

    private $user;

    private $acl;

    private $config;

    public function __construct(
        StreamService $streamService,
        Metadata $metadata,
        User $user,
        Acl $acl,
        Config $config
    ) {
        $this->streamService = $streamService;
        $this->metadata = $metadata;
        $this->user = $user;
        $this->acl = $acl;
        $this->config = $config;
    }

    public function process(Entity $entity, Params $params): void
    {
        $this->processIsFollowed($entity);
        $this->processFollowers($entity);
    }

    public function processIsFollowed(Entity $entity): void
    {
        if (!$entity->hasAttribute('isFollowed')) {
            return;
        }

        $isFollowed = $this->streamService->checkIsFollowed($entity);

        $entity->set('isFollowed', $isFollowed);
    }

    public function processFollowers(Entity $entity): void
    {
        if ($this->user->isPortal()) {
            return;
        }

        if (!$this->metadata->get(['scopes', $entity->getEntityType(), 'stream'])) {
            return;
        }

        if (!$this->acl->checkEntityStream($entity)) {
            return;
        }

        $limit = $this->config->get('recordFollowersLoadLimit') ?? self::FOLLOWERS_LIMIT;

        $data = $this->streamService->getEntityFollowers($entity, 0, $limit);

        if (!$data) {
            return;
        }

        $entity->set('followersIds', $data['idList']);
        $entity->set('followersNames', $data['nameMap']);
    }
}
