<?php
/************************************************************************
 * This file is part of Partners.
 *
 * Partners - Open Source CRM application.
 * Copyright (C) 2014-2019 Yuri Kuznetsov, Taras Machyshyn, Oleksiy Avramenko
 * Website: https://www.espocrm.com
 *
 * Partners is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Partners is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Partners. If not, see http://www.gnu.org/licenses/.
 *
 * The interactive user interfaces in modified source and object code versions
 * of this program must display Appropriate Legal Notices, as required under
 * Section 5 of the GNU General Public License version 3.
 *
 * In accordance with Section 7(b) of the GNU General Public License version 3,
 * these Appropriate Legal Notices must retain the display of the "Partners" word.
 ************************************************************************/

namespace Espo\Core\Webhook;

use Espo\Entities\Webhook;

class Sender
{
    const CONNECT_TIMEOUT = 5;

    const TIMEOUT = 10;

    protected $config;

    public function __construct(\Espo\Core\Utils\Config $config)
    {
        $this->config = $config;
    }

    public function send(Webhook $webhook, array $dataList) : int
    {
        $payload = json_encode($dataList);

        $signature = null;
        $secretKey = $webhook->get('secretKey');
        if ($secretKey) {
            $signature = $this->buildSignature($webhook, $payload, $secretKey);
        }

        $connectTimeout = $this->config->get('webhookConnectTimeout', self::CONNECT_TIMEOUT);
        $timeout = $this->config->get('webhookTimeout', self::TIMEOUT);

        $headerList = [];
        $headerList[] = 'Content-Type: application/json';
        $headerList[] = 'Content-Length: ' . strlen($payload);
        if ($signature) {
            $headerList[] = 'X-Signature: ' . $signature;
        }

        $handler = curl_init($webhook->get('url'));
        curl_setopt($handler, \CURLOPT_RETURNTRANSFER, true);
        curl_setopt($handler, \CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($handler, \CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($handler, \CURLOPT_HEADER, true);
        curl_setopt($handler, \CURLOPT_CUSTOMREQUEST, 'POST');
        curl_setopt($handler, \CURLOPT_CONNECTTIMEOUT, $connectTimeout);
        curl_setopt($handler, \CURLOPT_TIMEOUT, $timeout);
        curl_setopt($handler, \CURLOPT_HTTPHEADER, $headerList);
        curl_setopt($handler, \CURLOPT_POSTFIELDS, $payload);

        curl_exec($handler);

        $code = curl_getinfo($handler, \CURLINFO_HTTP_CODE);

        if (!is_numeric($code)) $code = 0;
        if (!is_int($code)) $code = intval($code);

        if ($errorNumber = curl_errno($handler)) {
            if (in_array($errorNumber, [\CURLE_OPERATION_TIMEDOUT, \CURLE_OPERATION_TIMEOUTED])) {
                $code = 408;
            }
        }

        curl_close($handler);

        return $code;
    }

    protected function buildSignature(Webhook $webhook, string $payload, string $secretKey)
    {
        return base64_encode($webhook->id . ':' . hash_hmac('sha256', $payload, $secretKey, true));
    }
}
