<?php
/***********************************************************************************
 * The contents of this file are subject to the Extension License Agreement
 * ("Agreement") which can be viewed at
 * https://www.espocrm.com/extension-license-agreement/.
 * By copying, installing downloading, or using this file, You have unconditionally
 * agreed to the terms and conditions of the Agreement, and You may not use this
 * file except in compliance with the Agreement. Under the terms of the Agreement,
 * You shall not license, sublicense, sell, resell, rent, lease, lend, distribute,
 * redistribute, market, publish, commercialize, or otherwise transfer rights or
 * usage to the software or any modified version or derivative work of the software
 * created by or for you.
 *
 * Copyright (C) 2015-2025 Letrium Ltd.
 *
 * License ID: a67be8f75bc1974a987315ddc1024a78
 ************************************************************************************/

namespace Espo\Modules\Sales\Controllers;

use Espo\Core\Api\Request;
use Espo\Core\Controllers\Record;
use Espo\Core\Exceptions\BadRequest;
use Espo\Core\Exceptions\Error;
use Espo\Core\Exceptions\Forbidden;
use Espo\Core\Exceptions\NotFound;
use Espo\Modules\Crm\Entities\Opportunity;
use Espo\Modules\Sales\Entities\Invoice as InvoiceEntity;
use Espo\Modules\Sales\Entities\Quote as QuoteEntity;
use Espo\Modules\Sales\Entities\SalesOrder as SalesOrderEntity;
use Espo\Modules\Sales\Tools\Quote\ConvertService;
use Espo\Modules\Sales\Tools\Invoice\EmailService;

/**
 * @noinspection PhpUnused
 */
class Invoice extends Record
{
    /**
     * @return array<string, mixed>
     * @throws BadRequest
     * @throws Forbidden
     * @throws NotFound
     */
    public function actionGetAttributesFromOpportunity(Request $request): array
    {
        $opportunityId = $request->getQueryParam('opportunityId');

        if (!$opportunityId) {
            throw new BadRequest();
        }

        return $this->injectableFactory
            ->create(ConvertService::class)
            ->getAttributes(InvoiceEntity::ENTITY_TYPE, Opportunity::ENTITY_TYPE, $opportunityId);
    }

    /**
     * @return array<string, mixed>
     * @throws BadRequest
     * @throws Forbidden
     * @throws NotFound
     */
    public function actionGetAttributesFromQuote(Request $request): array
    {
        $quoteId = $request->getQueryParam('quoteId');

        if (!$quoteId) {
            throw new BadRequest();
        }

        return $this->injectableFactory
            ->create(ConvertService::class)
            ->getAttributes(InvoiceEntity::ENTITY_TYPE, QuoteEntity::ENTITY_TYPE, $quoteId);
    }

    /**
     * @return array<string, mixed>
     * @throws BadRequest
     * @throws Forbidden
     * @throws NotFound
     */
    public function actionGetAttributesFromSalesOrder(Request $request): array
    {
        $salesOrderId = $request->getQueryParam('salesOrderId');

        if (!$salesOrderId) {
            throw new BadRequest();
        }

        return $this->injectableFactory
            ->create(ConvertService::class)
            ->getAttributes(InvoiceEntity::ENTITY_TYPE, SalesOrderEntity::ENTITY_TYPE, $salesOrderId);
    }

    /**
     * @return array<string, mixed>
     * @throws BadRequest
     * @throws Forbidden
     * @throws Error
     * @throws NotFound
     */
    public function postActionGetAttributesForEmail(Request $request): array
    {
        $data = $request->getParsedBody();

        if (empty($data->id) || empty($data->templateId)) {
            throw new BadRequest();
        }

        $format = $data->format ?? null;

        if ($format !== null && !is_string($format)) {
            throw new BadRequest("Bad format.");
        }

        return $this->injectableFactory
            ->create(EmailService::class)
            ->getAttributes($data->id, $data->templateId, null, $format);
    }
}
