/***********************************************************************************
 * The contents of this file are subject to the Extension License Agreement
 * ("Agreement") which can be viewed at
 * https://www.espocrm.com/extension-license-agreement/.
 * By copying, installing downloading, or using this file, You have unconditionally
 * agreed to the terms and conditions of the Agreement, and You may not use this
 * file except in compliance with the Agreement. Under the terms of the Agreement,
 * You shall not license, sublicense, sell, resell, rent, lease, lend, distribute,
 * redistribute, market, publish, commercialize, or otherwise transfer rights or
 * usage to the software or any modified version or derivative work of the software
 * created by or for you.
 *
 * Copyright (C) 2015-2025 Letrium Ltd.
 *
 * License ID: a67be8f75bc1974a987315ddc1024a78
 ************************************************************************************/

define('sales:views/purchase-order/modals/create-receipt', ['views/modal'], function (ModalView) {

    /**
     * @extends module:views/modal
     */
    class CreateReceiptModal extends ModalView {

        // language=Handlebars
        templateContent = `
            <div class="item-container record no-side-margin">
                <div class="record-container">{{{record}}}</div>
            </div>
        `

        /**
         * @private
         * @type {import('views/record/edit').default}
         */
        recordView

        /**
         * @private
         * @type {import('model').default}
         */
        receiptModel

        /**
         * @private
         * @type {Record.<string, *>}
         */
        attributes

        data() {
            return {
                collection: this.collection,
            };
        }

        setup() {
            this.headerText = this.translate('Create Receipt', 'labels', 'PurchaseOrder');

            this.buttonList = [
                {
                    name: 'create',
                    label: 'Create',
                    style: 'danger',
                    onClick: () => this.actionCreate(),
                },
                {
                    name: 'cancel',
                    label: 'Cancel',
                    onClick: () => this.close(),
                }
            ];

            this.wait(
                Promise
                    .all([
                        this.setupLayout(),
                        this.setupAttributes(),
                    ])
                    .then(() => this.setupModel())
                    .then(() => this.createRecordView())
            );
        }

        async setupAttributes() {
            const linkAttribute = Espo.Utils.lowerCaseFirst(this.model.entityType) + 'Id';

            const url = this.model.entityType === 'ReturnOrder' ?
               'ReceiptOrder/action/getAttributesFromReturnOrder' :
               'ReceiptOrder/action/getAttributesFromPurchaseOrder';

            this.attributes = await Espo.Ajax.getRequest(url, {[linkAttribute]: this.model.id});
        }

        setupLayout() {
            return new Promise(resolve => {
                this.getHelper().layoutManager.get('ReceiptOrder', 'detailCreateFromPurchaseOrder', layout => {
                    this.detailLayout = Espo.Utils.cloneDeep(layout);

                    this.detailLayout.push({
                        rows: [
                            [
                                {
                                    name: 'itemList',
                                    noLabel: true,
                                }
                            ]
                        ],
                    });

                    this.detailLayout.push({
                        name: 'receivedItems',
                        rows: [
                            [
                                {
                                    name: 'receivedItemList',
                                    view: 'sales:views/receipt-order/fields/received-item-list',
                                }
                            ]
                        ],
                    });

                    resolve();
                });
            });
        }

        async createRecordView() {
            this.recordView = await this.createView('record', 'views/record/edit-for-modal', {
                model: this.receiptModel,
                selector: `.record-container`,
                detailLayout: this.detailLayout,
            });

            this.controlReceiptItemsVisibility();

            this.listenTo(this.receiptModel, 'change:itemList', () => this.controlReceiptItemsVisibility());

            return this.recordView;
        }

        async setupModel() {
            const model = await this.getModelFactory().create('ReceiptOrder');

            model.populateDefaults();

            // @todo Revise.
            model.set(Espo.Utils.cloneDeep(this.attributes));

            model.set({
                assignedUserId: this.model.get('assignedUserId'),
                assignedUserName: this.model.get('assignedUserName'),
            });

            // @todo Revise.
            model.set(this.attributes);

            this.receiptModel = model;
        }

        actionCreate() {
            if (this.recordView.validate()) {
                return;
            }

            Espo.Ui.notify(' ... ');

            this.disableButton('create');

            this.receiptModel
                .save()
                .then(response => {
                    const url = '#ReceiptOrder/view/' + response.id;
                    const name = response.name;

                    const msg = this.translate('Created')  + '\n' + `[${name}](${url})`;

                    Espo.Ui.notify(msg, 'success', 4000, {suppress: true});

                    this.trigger('done');
                    this.close();
                })
                .catch(() => {
                    this.enableButton('create');
                });
        }

        /**
         * @private
         */
        controlReceiptItemsVisibility() {
            /** @type {{inventoryNumberType?: string|null, quantityReceived?: number|null}[]} */
            const items = this.receiptModel.attributes.itemList || [];

            for (const item of items) {
                if (item.inventoryNumberType && item.quantityReceived) {
                    this.recordView.showPanel('receivedItems');

                    return;
                }
            }

            this.recordView.hidePanel('receivedItems');
        }
    }

    return CreateReceiptModal;
});
