/***********************************************************************************
 * The contents of this file are subject to the Extension License Agreement
 * ("Agreement") which can be viewed at
 * https://www.espocrm.com/extension-license-agreement/.
 * By copying, installing downloading, or using this file, You have unconditionally
 * agreed to the terms and conditions of the Agreement, and You may not use this
 * file except in compliance with the Agreement. Under the terms of the Agreement,
 * You shall not license, sublicense, sell, resell, rent, lease, lend, distribute,
 * redistribute, market, publish, commercialize, or otherwise transfer rights or
 * usage to the software or any modified version or derivative work of the software
 * created by or for you.
 *
 * Copyright (C) 2015-2025 Letrium Ltd.
 *
 * License ID: e4c270586a0c8a9fda53bda910f357e0
 ************************************************************************************/

define('advanced:views/workflow/modals/add-action', ['views/modal'],
(/** typeof import('views/modal').default */Dep) => {

    return class extends Dep {

        // language=Handlebars
        templateContent = `
            <div class="margin-bottom-2x margin-top">
                <input
                    type="text"
                    maxlength="64"
                    placeholder="{{translate 'Search'}}"
                    data-name="quick-search"
                    class="form-control"
                    spellcheck="false"
                >
            </div>
            <ul class="list-group list-group-panel array-add-list-group no-side-margin">
            {{#each itemList}}
                <li class="list-group-item" data-name="{{name}}">
                    <a
                        role="button"
                        tabindex="0"
                        class="text-bold"
                        data-action="add"
                        data-name="{{name}}"
                    >{{label}}</a>
                </li>
            {{/each}}
            </ul>
            <div class="no-data hidden">{{translate 'No Data'}}</div>
        `

        backdrop = true

        data() {
            return {
                itemList: this.actionDataList,
            };
        }

        setup() {
            this.headerText = this.translate('Add Action', 'labels', 'Workflow');

            this.addActionHandler('add', (e, target) => this.actionAdd(target.dataset.name));

            this.addHandler('keyup', 'input[data-name="quick-search"]', (e, /** HTMLInputElement */target) => {
                return this.processQuickSearch(target.value);
            });

            /** @type {string[]} */
            this.actionList = this.options.actionList;

            /** @type {{name: string, label: string}[]} */
            this.actionDataList = this.getDataList();
        }

        afterRender() {
            if (!this.element) {
                return;
            }

            /** @type {HTMLElement} */
            this.noDataElement = this.element.querySelector('.no-data');
            /** @type {HTMLInputElement} */
            this.quickSearchInputElement = this.element.querySelector('input[data-name="quick-search"]');
            /** @type {HTMLElement[]} */
            this.itemElements = [...this.element.querySelectorAll('ul > li.list-group-item')];

            setTimeout(() => this.quickSearchInputElement.focus(), 100);
        }

        getDataList() {
            return this.actionList
                .map(it => (
                    {
                        name: it,
                        label: this.translate(it, 'actionTypes', 'Workflow')
                    }
                ))
                .sort((a, b) => a.label.localeCompare(b.label));
        }

        /**
         * @private
         * @param {string} name
         */
        actionAdd(name) {
            this.trigger('add', name);

            this.close();
        }

        /**
         * @private
         * @param {string} text
         */
        processQuickSearch(text) {
            text = text.trim().toLowerCase();

            this.noDataElement.classList.add('hidden');

            if (!text) {
                this.itemElements.forEach(element => element.classList.remove('hidden'));

                return;
            }

            const matchedList = [];

            this.actionDataList.forEach(item => {
                const label = item.label.toLowerCase();

                for (const word of label.split(' ')) {
                    const matched = word.indexOf(text) === 0;

                    if (matched) {
                        matchedList.push(item);

                        return;
                    }
                }
            });

            if (matchedList.length === 0) {
                this.itemElements.forEach(element => element.classList.add('hidden'));

                this.noDataElement.classList.remove('hidden');

                return;
            }

            this.actionDataList.forEach(item => {
                const element = this.itemElements.find(element => element.dataset.name === item.name);

                if (!matchedList.includes(item)) {
                    element.classList.add('hidden');
                } else {
                    element.classList.remove('hidden');
                }
            });
        }
    }
});
