<?php
/***********************************************************************************
 * The contents of this file are subject to the Extension License Agreement
 * ("Agreement") which can be viewed at
 * https://www.espocrm.com/extension-license-agreement/.
 * By copying, installing downloading, or using this file, You have unconditionally
 * agreed to the terms and conditions of the Agreement, and You may not use this
 * file except in compliance with the Agreement. Under the terms of the Agreement,
 * You shall not license, sublicense, sell, resell, rent, lease, lend, distribute,
 * redistribute, market, publish, commercialize, or otherwise transfer rights or
 * usage to the software or any modified version or derivative work of the software
 * created by or for you.
 *
 * Copyright (C) 2024-2025 Letrium Ltd.
 *
 * License ID: f27e70ce6801a13265271f5669c8bc5c
 ************************************************************************************/

namespace Espo\Modules\Project\Classes\Acl\ProjectTask\LinkCheckers;

use Espo\Core\Acl;
use Espo\Core\Acl\LinkChecker;
use Espo\Entities\User;
use Espo\Modules\Project\Entities\Project;
use Espo\Modules\Project\Entities\ProjectRole;
use Espo\Modules\Project\Entities\ProjectTask;
use Espo\Modules\Project\Tools\Project\MemberRoleProvider;
use Espo\ORM\Entity;
use Espo\ORM\EntityManager;

/**
 * @implements LinkChecker<ProjectTask, User>
 */
class Owner implements LinkChecker
{
    public function __construct(
        private EntityManager $entityManager,
        private Acl $acl,
        private MemberRoleProvider $memberRoleProvider,
    ) {}

    public function check(User $user, Entity $entity, Entity $foreignEntity): bool
    {
        return $this->checkOwner($user, $entity);
    }

    private function checkOwner(User $user, ProjectTask $entity): bool
    {
        if (!$entity->getProjectId()) {
            return false;
        }

        if (!$entity->getOwnerId()) {
            return true;
        }

        $project = $this->entityManager->getRDBRepositoryByClass(Project::class)->getById($entity->getProjectId());

        if (!$project) {
            return false;
        }

        if (
            $entity->getOwnerId() === $user->getId() &&
            $this->acl->checkEntityEdit($project)
        ) {
            return true;
        }

        $isRelated = $this->entityManager
            ->getRelation($project, 'members')
            ->isRelatedById($entity->getOwnerId());

        if (!$isRelated) {
            return false;
        }

        if ($user->isAdmin()) {
            return true;
        }

        $memberRole = $this->memberRoleProvider->get($user, $project->getId());

        if (!$memberRole) {
            return false;
        }

        if (
            $this->acl->checkEntityEdit($project) ||
            $memberRole->record->getTaskAssignment() === ProjectRole::LEVEL_ALL
        ) {
            return true;
        }

        return $entity->getOwnerId() === $user->getId();
    }
}
