<?php
/***********************************************************************************
 * The contents of this file are subject to the Extension License Agreement
 * ("Agreement") which can be viewed at
 * https://www.espocrm.com/extension-license-agreement/.
 * By copying, installing downloading, or using this file, You have unconditionally
 * agreed to the terms and conditions of the Agreement, and You may not use this
 * file except in compliance with the Agreement. Under the terms of the Agreement,
 * You shall not license, sublicense, sell, resell, rent, lease, lend, distribute,
 * redistribute, market, publish, commercialize, or otherwise transfer rights or
 * usage to the software or any modified version or derivative work of the software
 * created by or for you.
 *
 * Copyright (C) 2024-2025 Letrium Ltd.
 *
 * License ID: f27e70ce6801a13265271f5669c8bc5c
 ************************************************************************************/

namespace Espo\Modules\Project\Classes\Acl\Common\AccessCheckers;

use Espo\Core\AclManager;
use Espo\Entities\User;
use Espo\ORM\Entity;
use Espo\Core\Utils\Metadata;
use Espo\Core\Acl\DefaultAccessChecker;
use Espo\Core\Acl\Traits\DefaultAccessCheckerDependency;
use Espo\Core\Acl\AccessEntityCreateChecker;
use Espo\Core\Acl\AccessEntityReadChecker;
use Espo\Core\Acl\AccessEntityEditChecker;
use Espo\Core\Acl\AccessEntityDeleteChecker;
use Espo\Core\Acl\ScopeData;
use Espo\ORM\EntityManager;
use LogicException;

/**
 * @implements AccessEntityCreateChecker<Entity>
 * @implements AccessEntityReadChecker<Entity>
 * @implements AccessEntityEditChecker<Entity>
 * @implements AccessEntityDeleteChecker<Entity>
 */
class Foreign implements

    AccessEntityCreateChecker,
    AccessEntityReadChecker,
    AccessEntityEditChecker,
    AccessEntityDeleteChecker
{
    use DefaultAccessCheckerDependency;

    public function __construct(
        private Metadata $metadata,
        private EntityManager $entityManager,
        private AclManager $aclManager,
        DefaultAccessChecker $defaultAccessChecker,
    ) {
        $this->defaultAccessChecker = $defaultAccessChecker;
    }

    private function getForeignEntity(Entity $entity): ?Entity
    {
        $entityType = $entity->getEntityType();

        $link = $this->metadata->get(['aclDefs', $entityType, 'link']);

        if (!$link) {
            throw new LogicException("No `link` in aclDefs for $entityType.");
        }

        if ($entity->isNew()) {
            $foreignEntityType = $this->entityManager
                ->getDefs()
                ->getEntity($entityType)
                ->getRelation($link)
                ->getForeignEntityType();

            /** @var ?string $id */
            $id = $entity->get($link . 'Id');

            if (!$id) {
                return null;
            }

            return $this->entityManager->getEntityById($foreignEntityType, $id);
        }

        return $this->entityManager
            ->getRDBRepository($entityType)
            ->getRelation($entity, $link)
            ->findOne();
    }

    public function checkCreate(User $user, ScopeData $data): bool
    {
        if ($this->defaultAccessChecker->checkCreate($user, $data)) {
            return true;
        }

        // Allow 'create' if not allowed in roles access can be granted via membership.
        return $this->defaultAccessChecker->check($user, $data);
    }

    public function checkRead(User $user, ScopeData $data): bool
    {
        if ($this->defaultAccessChecker->checkRead($user, $data)) {
            return true;
        }

        // Allow 'read' if not allowed in roles access can be granted via membership.
        return $this->defaultAccessChecker->check($user, $data);
    }

    public function checkEdit(User $user, ScopeData $data): bool
    {
        if ($this->defaultAccessChecker->checkEdit($user, $data)) {
            return true;
        }

        // Allow 'edit' if not allowed in roles access can be granted via membership.
        return $this->defaultAccessChecker->check($user, $data);
    }

    public function checkDelete(User $user, ScopeData $data): bool
    {
        if ($this->defaultAccessChecker->checkEdit($user, $data)) {
            return true;
        }

        return $this->defaultAccessChecker->check($user, $data);
    }

    public function checkEntityCreate(User $user, Entity $entity, ScopeData $data): bool
    {
        $foreign = $this->getForeignEntity($entity);

        if (!$foreign) {
            return false;
        }

        return $this->aclManager->checkEntityEdit($user, $foreign);
    }

    public function checkEntityRead(User $user, Entity $entity, ScopeData $data): bool
    {
        $foreign = $this->getForeignEntity($entity);

        if (!$foreign) {
            return false;
        }

        return $this->aclManager->checkEntityRead($user, $foreign);
    }

    public function checkEntityEdit(User $user, Entity $entity, ScopeData $data): bool
    {
        $foreign = $this->getForeignEntity($entity);

        if (!$foreign) {
            return false;
        }

        return $this->aclManager->checkEntityEdit($user, $foreign);
    }

    public function checkEntityDelete(User $user, Entity $entity, ScopeData $data): bool
    {
        $foreign = $this->getForeignEntity($entity);

        if (!$foreign) {
            if ($user->isAdmin()) {
                return true;
            }

            return false;
        }

        if ($this->aclManager->checkEntityEdit($user, $foreign)) {
            return true;
        }

        return $this->aclManager->checkEntityDelete($user, $foreign);
    }
}
