/***********************************************************************************
 * The contents of this file are subject to the Extension License Agreement
 * ("Agreement") which can be viewed at
 * https://www.espocrm.com/extension-license-agreement/.
 * By copying, installing downloading, or using this file, You have unconditionally
 * agreed to the terms and conditions of the Agreement, and You may not use this
 * file except in compliance with the Agreement. Under the terms of the Agreement,
 * You shall not license, sublicense, sell, resell, rent, lease, lend, distribute,
 * redistribute, market, publish, commercialize, or otherwise transfer rights or
 * usage to the software or any modified version or derivative work of the software
 * created by or for you.
 *
 * Copyright (C) 2015-2025 Letrium Ltd.
 *
 * License ID: e4c270586a0c8a9fda53bda910f357e0
 ************************************************************************************/

define('advanced:views/bpmn-flowchart/fields/flowchart', ['views/fields/base'],
function (/** typeof import('views/fields/base').default */Dep) {

    return class FlowchartFieldView extends Dep {

        detailTemplate = 'advanced:bpmn-flowchart/fields/flowchart/detail'
        editTemplate = 'advanced:bpmn-flowchart/fields/flowchart/edit'

        height = 500
        inlineEditDisabled = true
        dataAttribute = 'data'


        /**
         * @private
         * @type {number}
         */
        undoStackLength = 30

        events = {
            /** @this {FlowchartFieldView} */
            'click .action[data-action="setStateCreateFigure"]': function (e) {
                const type = $(e.currentTarget).data('name');

                this.setStateCreateFigure(type);
            },
            /** @this {FlowchartFieldView} */
            'click .action[data-action="resetState"]': function () {
                this.resetState(true);
            },
            /** @this {FlowchartFieldView} */
            'click .action[data-action="setStateCreateFlow"]': function () {
                this.setStateCreateFlow();
            },
            /** @this {FlowchartFieldView} */
            'click .action[data-action="setStateRemove"]': function () {
                this.setStateRemove();
            },
            /** @this {FlowchartFieldView} */
            'click .action[data-action="apply"]': function () {
                this.apply();
            },
            /** @this {FlowchartFieldView} */
            'click .action[data-action="zoomIn"]': function () {
                this.zoomIn();
            },
            /** @this {FlowchartFieldView} */
            'click .action[data-action="zoomOut"]': function () {
                this.zoomOut();
            },
            /** @this {FlowchartFieldView} */
            'click .action[data-action="switchFullScreenMode"]': function (e) {
                e.preventDefault();

                if (this.isFullScreenMode) {
                    this.unsetFullScreenMode();
                } else {
                    this.setFullScreenMode();
                }
            },
        }

        getAttributeList() {
            return [this.dataAttribute];
        }

        data() {
            const data = super.data();

            data.heightString = this.height.toString() + 'px';

            if (this.mode  === 'edit') {
                data.elementEventDataList = this.elementEventDataList;
                data.elementGatewayDataList = this.elementGatewayDataList;
                data.elementTaskDataList = this.elementTaskDataList;
                data.currentElement = this.currentElement;
            }

            return data;
        }

        setup() {
            super.setup();

            this.undoStack = [];
            this.redoStack = [];

            this.addActionHandler('moveToCenter', () => this.moveToCenter());
            this.addActionHandler('undo', () => this.undo());
            this.addActionHandler('redo', () => this.redo());

            const startColor = '#69a345';
            const intermediateColor = '#5d86b0';
            const endColor = '#b34646';

            this.elementEventList = [
                'eventStart',
                'eventStartConditional',
                'eventStartTimer',
                'eventStartError',
                'eventStartEscalation',
                'eventStartSignal',
                'eventStartCompensation',
                'eventIntermediateConditionalCatch',
                'eventIntermediateTimerCatch',
                'eventIntermediateSignalCatch',
                'eventIntermediateMessageCatch',
                'eventIntermediateEscalationThrow',
                'eventIntermediateSignalThrow',
                'eventIntermediateCompensationThrow',
                'eventEnd',
                'eventEndTerminate',
                'eventEndError',
                'eventEndEscalation',
                'eventEndSignal',
                'eventEndCompensation',
                'eventIntermediateErrorBoundary',
                'eventIntermediateConditionalBoundary',
                'eventIntermediateTimerBoundary',
                'eventIntermediateEscalationBoundary',
                'eventIntermediateSignalBoundary',
                'eventIntermediateMessageBoundary',
                'eventIntermediateCompensationBoundary',
            ];

            this.elementGatewayList = [
                'gatewayExclusive',
                'gatewayInclusive',
                'gatewayParallel',
                'gatewayEventBased',
            ];

            this.elementTaskList = [
                'task',
                'taskSendMessage',
                'taskUser',
                'taskScript',
                '_divider',
                'subProcess',
                'eventSubProcess',
                'callActivity',
            ];

            this.elementEventDataList = [
                {name: 'eventStart', color: startColor},
                {name: 'eventStartConditional', color: startColor},
                {name: 'eventStartTimer', color: startColor},
                {name: 'eventStartError', color: startColor},
                {name: 'eventStartEscalation', color: startColor},
                {name: 'eventStartSignal', color: startColor},
                {name: 'eventStartCompensation', color: startColor},
                {name: '_divider', color: null},
                {name: 'eventIntermediateConditionalCatch', color: intermediateColor},
                {name: 'eventIntermediateTimerCatch', color: intermediateColor},
                {name: 'eventIntermediateSignalCatch', color: intermediateColor},
                {name: 'eventIntermediateMessageCatch', color: intermediateColor},
                {name: '_divider', color: null},
                {name: 'eventIntermediateEscalationThrow', color: intermediateColor},
                {name: 'eventIntermediateSignalThrow', color: intermediateColor},
                {name: 'eventIntermediateCompensationThrow', color: intermediateColor},
                {name: '_divider', color: null},
                {name: 'eventEnd', color: endColor},
                {name: 'eventEndTerminate', color: endColor},
                {name: 'eventEndError', color: endColor},
                {name: 'eventEndEscalation', color: endColor},
                {name: 'eventEndSignal', color: endColor},
                {name: 'eventEndCompensation', color: endColor},
                {name: '_divider', color: null},
                {name: 'eventIntermediateErrorBoundary', color: intermediateColor},
                {name: 'eventIntermediateConditionalBoundary', color: intermediateColor},
                {name: 'eventIntermediateTimerBoundary', color: intermediateColor},
                {name: 'eventIntermediateEscalationBoundary', color: intermediateColor},
                {name: 'eventIntermediateSignalBoundary', color: intermediateColor},
                {name: 'eventIntermediateMessageBoundary', color: intermediateColor},
                {name: 'eventIntermediateCompensationBoundary', color: intermediateColor},
            ];

            this.elementGatewayDataList = [];
            this.elementGatewayList.forEach(item => {
                this.elementGatewayDataList.push({name: item});
            });

            this.elementTaskDataList = [];
            this.elementTaskList.forEach(item => {
                this.elementTaskDataList.push({name: item});
            });

            this.dataHelper = {
                getAllDataList: () => {
                    return this.getAllDataList();
                },
                getElementData: (id) => {
                    return this.getElementData(id);
                },
            };

            this.wait(true);

            Espo.loader.require('lib!client/custom/modules/advanced/lib/espo-bpmn.js', () => this.wait(false));

            this.on('inline-edit-off', () => {
                this.currentState = null;
                this.currentElement = null;
            });

            const data = Espo.Utils.cloneDeep(this.model.get(this.dataAttribute) || {});

            this.dataList = data.list || [];
            this.createdEntitiesData = data.createdEntitiesData || {};

            this.listenTo(this.model, 'change:' + this.dataAttribute, (model, value, o) => {
                if (o.ui) {
                    return;
                }

                const data = Espo.Utils.cloneDeep(this.model.get(this.dataAttribute) || {});

                this.dataList = data.list || [];
            });

            this.on('render', () => {
                if (this.canvas) {
                    this.offsetX = this.canvas.offsetX;
                    this.offsetY = this.canvas.offsetY;
                    this.scaleRatio = this.canvas.scaleRatio;
                }
            });

            this.offsetX = null;
            this.offsetY = null;
            this.scaleRatio = null;
        }

        prepare() {
            if (this.mode === this.MODE_EDIT) {
                this.undoStack = [];
                this.redoStack = [];

                this.addDataToUndoStack();
            }

            return super.prepare();
        }

        afterRender() {
            this.$groupContainer = this.$el.find('.flowchart-group-container');
            this.$container = this.$el.find('.flowchart-container');

            if (this.isFullScreenMode) {
                this.setFullScreenMode();
            }

            if (this.isEditMode()) {
                this.controlUndoActions();
            }

            const dataDefaults = {};
            const elements = this.getMetadata().get(['clientDefs', 'BpmnFlowchart', 'elements']) || {};

            for (const type in elements) {
                if ('defaults' in elements[type]) {
                    dataDefaults[type] = Espo.Utils.cloneDeep(elements[type].defaults);
                }
            }

            dataDefaults.subProcess = dataDefaults.subProcess || {};
            dataDefaults.subProcess.targetType = this.model.get('targetType');

            dataDefaults.eventSubProcess = dataDefaults.eventSubProcess || {};
            dataDefaults.eventSubProcess.targetType = this.model.get('targetType');

            const o = {
                canvasWidth: '100%',
                canvasHeight: '100%',
                dataDefaults: dataDefaults,
                events: {
                    change: () => {
                        if (this.mode === this.MODE_EDIT) {
                            this.trigger('change');

                            this.addDataToUndoStack();
                        }
                    },
                    resetState: () => {
                        this.currentElement = null;
                        this.currentState = null;

                        this.resetTool(true);
                    },
                    figureLeftClick: (e) => {
                        const id = e.figureId;

                        if (!id) {
                            return;
                        }

                        this.openElement(id);
                    },
                    removeFigure: (e) => {
                        this.onRemoveElement(e.figureId);
                        this.trigger('change');

                        this.addDataToUndoStack();
                    },
                    createFigure: (e) => {
                        this.onCreateElement(e.figureId/*, e.figureData*/);

                        this.addDataToUndoStack();
                    },
                },
                isReadOnly: this.mode !== this.MODE_EDIT,
                scaleDisabled: true,
                isEventSubProcess: this.isEventSubProcess,
            };

            if (this.getThemeManager().getParam('isDark')) {
                this.applyDarkColorsToCanvasOptions(o);
            }

            if (this.offsetX !== null) {
                o.offsetX = this.offsetX;
            }

            if (this.offsetY !== null) {
                o.offsetY = this.offsetY;
            }

            if (this.scaleRatio !== null) {
                o.scaleRatio = this.scaleRatio;
            }

            const containerElement = this.$container.get(0);

            this.canvas = new window.EspoBpmn.Canvas(containerElement, o, this.dataList);

            if (this.mode === this.MODE_EDIT) {
                if (this.currentState) {
                    if (this.currentState === 'createFigure') {
                        this.setStateCreateFigure(this.currentElement);
                    } else {
                        const methodName = 'setState' + Espo.Utils.upperCaseFirst(this.currentState);

                        this[methodName]();
                    }
                } else {
                    this.resetTool(true);
                }
            }
        }

        openElement(id) {
            const elementData = this.getElementData(id);

            if (!elementData) {
                return;
            }

            const parentSubProcessData = this.getParentSubProcessData(id);

            let targetType = this.model.get('targetType');

            if (parentSubProcessData) {
                targetType = parentSubProcessData.targetType || targetType;
            }

            let isInSubProcess = this.isSubProcess;

            if (!isInSubProcess) {
                isInSubProcess = !!parentSubProcessData;
            }

            if (!isInSubProcess) {
                isInSubProcess = !!this.model.get('parentProcessId');
            }

            if (this.mode === this.MODE_DETAIL) {
                this.createView('modalView', 'advanced:views/bpmn-flowchart/modals/element-detail', {
                    elementData: elementData,
                    targetType: targetType,
                    flowchartDataList: this.dataList,
                    flowchartModel: this.model,
                    flowchartCreatedEntitiesData: this.createdEntitiesData,
                    dataHelper: this.dataHelper,
                    isInSubProcess: isInSubProcess,
                    isInSubProcess2: !!parentSubProcessData,
                }, (view) => {
                    view.render();

                    this.listenToOnce(view, 'remove', () => {
                        this.clearView('modalView');
                    });
                });

                return;
            }

            if (this.mode === this.MODE_EDIT) {
                this.createView('modalEdit', 'advanced:views/bpmn-flowchart/modals/element-edit', {
                    elementData: elementData,
                    targetType: targetType,
                    flowchartDataList: this.dataList,
                    flowchartModel: this.model,
                    flowchartCreatedEntitiesData: this.createdEntitiesData,
                    dataHelper: this.dataHelper,
                    isInSubProcess: isInSubProcess,
                }, (view) => {
                    view.render();

                    this.listenTo(view, 'apply', (data) => {
                        for (const attr in data) {
                            elementData[attr] = data[attr];
                        }

                        if ('defaultFlowId' in data) {
                            this.updateDefaultFlow(id);
                        }

                        if ('actionList' in data) {
                            this.updateCreatedEntitiesData(id, data.actionList, targetType);
                        } else if (elementData.type === 'taskUser') {
                            this.updateCreatedEntitiesDataUserTask(id, data);
                        } else if (elementData.type === 'taskSendMessage') {
                            this.updateCreatedEntitiesDataSendMessageTask(id, data);
                        }

                        if (parentSubProcessData && parentSubProcessData.type === 'eventSubProcess') {
                            if ((elementData.type || '').indexOf('eventStart') === 0) {
                                this.updateEventSubProcessStartData(parentSubProcessData.id, data);
                            }
                        }

                        this.trigger('change');

                        this.addDataToUndoStack();

                        view.remove();

                        this.reRender();
                    });

                    this.listenToOnce(view, 'remove', () => {
                        this.clearView('modalEdit');
                    });
                });
            }
        }

        updateCreatedEntitiesDataUserTask(id, data) {
            const numberId = (id in this.createdEntitiesData) ?
                this.createdEntitiesData[id].numberId :
                this.getNextCreatedEntityNumberId('BpmnUserTask');

            this.createdEntitiesData[id] = {
                elementId: id,
                actionId: null,
                entityType: 'BpmnUserTask',
                numberId: numberId,
                text: data.text || null,
            };
        }

        updateCreatedEntitiesDataSendMessageTask(id, data) {
            if (data.messageType === 'Email' && !data.doNotStore) {
                const numberId = (id in this.createdEntitiesData) ?
                    this.createdEntitiesData[id].numberId :
                    this.getNextCreatedEntityNumberId('Email');

                this.createdEntitiesData[id] = {
                    elementId: id,
                    actionId: null,
                    entityType: 'Email',
                    numberId: numberId,
                    text: data.text || null,
                };

                return;
            }

            delete this.createdEntitiesData[id];
        }

        removeCreatedEntitiesDataUserTask(id) {
            delete this.createdEntitiesData[id];
        }

        updateCreatedEntitiesData(id, actionList, targetType) {
            const toRemoveEIdList = [];

            for (const eId in this.createdEntitiesData) {
                const item = this.createdEntitiesData[eId];

                if (item.elementId === id) {
                    let isMet = false;

                    actionList.forEach(actionItem => {
                        if (item.actionId === actionItem.id) {
                            isMet = true;
                        }
                    });

                    if (!isMet) {
                        toRemoveEIdList.push(eId);
                    }
                }
            }

            toRemoveEIdList.forEach((eId) => {
                delete this.createdEntitiesData[eId];
            });

            actionList.forEach(item => {
                if (!~['createRelatedEntity', 'createEntity'].indexOf(item.type)) {
                    return;
                }

                const eId = id + '_' + item.id;

                let entityType;
                let link = null;

                if (item.type === 'createEntity') {
                    entityType = item.entityType;
                } else if (item.type === 'createRelatedEntity') {
                    link = item.link;
                    targetType = targetType || this.model.get('targetType');
                    entityType = this.getMetadata().get(['entityDefs', targetType, 'links', item.link, 'entity']);
                }

                if (!entityType) {
                    return;
                }

                const numberId = (eId in this.createdEntitiesData) ?
                    this.createdEntitiesData[eId].numberId :
                    this.getNextCreatedEntityNumberId(entityType);

                this.createdEntitiesData[eId] = {
                    elementId: id,
                    actionId: item.id,
                    link: link,
                    entityType: entityType,
                    numberId: numberId,
                };
            });
        }

        getNextCreatedEntityNumberId(entityType) {
            let numberId = 0;

            for (const eId in this.createdEntitiesData) {
                const item = this.createdEntitiesData[eId];

                if (entityType === item.entityType) {
                    if ('numberId' in item) {
                        numberId = item.numberId + 1;
                    }
                }
            }

            return numberId;
        }

        updateDefaultFlow(id) {
            const data = this.getElementData(id);
            const flowIdItemList = this.getElementFlowIdList(id);

            flowIdItemList.forEach(flowId => {
                const flowData = this.getElementData(flowId);

                if (!flowData) {
                    return;
                }

                flowData.isDefault = data.defaultFlowId === flowId;
            });
        }

        getElementFlowIdList(id) {
            const idList = [];

            this.getAllDataList().forEach(item => {
                if (item.type !== 'flow') {
                    return;
                }

                if (item.startId === id && item.endId) {
                    const endItem = this.getElementData(item.endId);

                    if (!endItem) {
                        return;
                    }

                    idList.push(item.id);
                }
            });

            return idList;
        }

        getElementData(id) {
            const o = {};

            this._findElementData(id, this.dataList, o);

            return o.data || null;
        }

        _findElementData(id, dataList, o) {
            for (let i = 0; i < dataList.length; i++) {
                if (dataList[i].id === id) {
                    o.data = dataList[i];

                    return;
                }

                if (dataList[i].type === 'subProcess' || dataList[i].type === 'eventSubProcess') {
                    this._findElementData(id, dataList[i].dataList || [], o);

                    if (o.data) {
                        return;
                    }
                }
            }
        }

        /*getParentSubProcessId(id) {
            const o = {};

            this._findParentSubProcessId(id, this.dataList, o);

            return o.id || null;
        }*/

        /*_findParentSubProcessId(id, dataList, o, parentId) {
            for (let i = 0; i < dataList.length; i++) {
                if (dataList[i].id === id) {
                    o.id = parentId;

                    return;
                }

                if (dataList[i].type === 'subProcess' || dataList[i].type === 'eventSubProcess') {
                    this._findParentSubProcessId(id, dataList[i].dataList || [], o, dataList[i].id);

                    if (o.id) {
                        return;
                    }
                }
            }
        }*/

        getParentSubProcessData(id) {
            const o = {};

            this._findParentSubProcessData(id, this.dataList, o);

            return o.data || null;
        }

        _findParentSubProcessData(id, dataList, o, parentData) {
            for (let i = 0; i < dataList.length; i++) {
                if (dataList[i].id === id) {
                    o.data = parentData;

                    return;
                }

                if (dataList[i].type === 'subProcess' || dataList[i].type === 'eventSubProcess') {
                    this._findParentSubProcessData(id, dataList[i].dataList || [], o, dataList[i]);

                    if (o.data) {
                        return;
                    }
                }
            }
        }

        updateEventSubProcessStartData(id, data) {
            const item = this.getElementData(id);

            item.eventStartData = _.extend(item.eventStartData, Espo.Utils.cloneDeep(data));
        }

        resetTool(isHandTool) {
            this.$el.find('.action[data-action="setStateCreateFigure"] span').addClass('hidden');
            this.$el.find('.button-container .btn').removeClass('active');

            if (isHandTool) {
                this.$el.find('.button-container .btn[data-action="resetState"]').addClass('active');
            }
        }

        setStateCreateFigure(type) {
            this.currentElement = type;

            this.currentState = 'createFigure';
            this.canvas.setState('createFigure', {type: type});

            this.resetTool();

            this.$el
                .find('.action[data-action="setStateCreateFigure"][data-name="'+type+'"] span')
                .removeClass('hidden');

            if (~this.elementEventList.indexOf(type)) {
                this.$el.find('.button-container .btn.add-event-element').addClass('active');
            }
            else if (~this.elementGatewayList.indexOf(type)) {
                this.$el.find('.button-container .btn.add-gateway-element').addClass('active');
            }
            else if (~this.elementTaskList.indexOf(type)) {
                this.$el.find('.button-container .btn.add-task-element').addClass('active');
            }
        }

        setStateCreateFlow() {
            this.resetState();

            this.currentState = 'createFlow';
            this.canvas.setState('createFlow');

            this.$el.find('.button-container .btn[data-action="setStateCreateFlow"]').addClass('active');
        }

        setStateRemove() {
            this.resetState();

            this.currentState = 'remove';
            this.canvas.setState('remove');

            this.$el.find('.button-container .btn[data-action="setStateRemove"]').addClass('active');
        }

        resetState(isHandTool) {
            this.canvas.resetState();

            this.currentState = null;
            this.currentElement = null;

            this.resetTool(isHandTool);
        }

        getAllDataList() {
            const list = [];

            this._populateAllList(this.dataList, list);

            return list;
        }

        _populateAllList(dataList, list) {
            for (let i = 0; i < dataList.length; i++) {
                list.push(dataList[i]);

                if (dataList[i].type === 'subProcess' || dataList[i].type === 'eventSubProcess') {
                    this._populateAllList(dataList[i].dataList || [], list);
                }
            }
        }

        onCreateElement(id/*, data*/) {
            const item = this.getElementData(id);

            if (item.type === 'taskUser') {
                this.updateCreatedEntitiesDataUserTask(id, item);
            } else if (item.type === 'taskSendMessage') {
                this.updateCreatedEntitiesDataSendMessageTask(id, item);
            }
        }

        onRemoveElement(id) {
            this.getAllDataList().forEach(item => {
                if (item.defaultFlowId === id) {
                    item.defaultFlowId = null;
                }

                if (item.flowList) {
                    const flowList = item.flowList;
                    const flowListCopy = [];

                    let isMet = false;

                    flowList.forEach(flowItem => {
                        if (flowItem.id === id) {
                            isMet = true;

                            return;
                        }

                        flowListCopy.push(flowItem);
                    });

                    if (isMet) {
                        item.flowList = flowListCopy;
                    }
                }
            });

            this.updateCreatedEntitiesData(id, []);
            this.removeCreatedEntitiesDataUserTask(id);
        }

        setFullScreenMode() {
            this.isFullScreenMode = true;

            let color = null;
            let $ref = this.$groupContainer;

            while (1) {
                color = window.getComputedStyle($ref.get(0), null).getPropertyValue('background-color');

                if (color && color !== 'rgba(0, 0, 0, 0)') {
                    break;
                }

                $ref = $ref.parent();

                if (!$ref.length) {
                    break;
                }
            }

            this.$groupContainer.css({
                width: '100%',
                height: '100%',
                position: 'fixed',
                top: 0,
                zIndex: 1050,
                left: 0,
                backgroundColor: color,
            });

            this.$container.css('height', '100%');
            this.$el.find('button[data-action="apply"]').removeClass('hidden');

            this.canvas.params.scaleDisabled = false;

            this.$groupContainer.addClass('fullscreen');
        }

        unsetFullScreenMode() {
            this.isFullScreenMode = false;

            this.$groupContainer.css({
                width: '',
                height: '',
                position: 'static',
                top: '',
                left: '',
                zIndex: '',
                backgroundColor: '',
            });

            this.$container.css('height', this.height.toString() + 'px');

            this.$groupContainer.removeClass('fullscreen');

            this.$el.find('button[data-action="apply"]').addClass('hidden');

            this.canvas.params.scaleDisabled = true;
        }

        apply () {
            if (this.model.isNew() && !this.model.attributes.name) {
                this.model.set('name', 'Unnamed');
            }

            this.model
                .save()
                .then(() => {
                    Espo.Ui.success(this.translate('Saved'));
                });
        }

        zoomIn() {
            this.canvas.scaleCentered(2);
        }

        zoomOut() {
            this.canvas.scaleCentered(-2);
        }

        fetch() {
            const fieldData = {};

            fieldData.list = Espo.Utils.cloneDeep(this.dataList);
            fieldData.createdEntitiesData = Espo.Utils.cloneDeep(this.createdEntitiesData);

            const data = {};

            data[this.dataAttribute] = fieldData;

            return data;
        }

        applyDarkColorsToCanvasOptions(o) {
            o.textColor = this.getThemeManager().getParam('textColor') || '#fff';
            o.strokeColor = '#8a8f89';
            o.rectangleExpandedFillColor = '#242424';
            o.taskStrokeColor = o.strokeColor;
            o.taskFillColor = '#1a1a1a';
            o.eventStartFillColor = '#70995e';
            o.eventStartStrokeColor = '#426e26';
            o.gatewayStrokeColor = '#7e7437';
            o.gatewayFillColor = '#afa152';
            o.eventEndFillColor = '#ab5b5b';
            o.eventEndStrokeColor = '#6a3b3b';
            o.eventIntermediateFillColor = '#6c88b3';
            o.eventIntermediateStrokeColor = '#435d87';
        }

        moveToCenter() {
            this.offsetX = 0;
            this.offsetY = 0;

            this.canvas.moveTo({x: 0, y: 0});
        }

        controlUndoActions() {
            if (!this.element) {
                return;
            }

            const undo = this.element.querySelector('[data-action="undo"]');
            const redo = this.element.querySelector('[data-action="redo"]');

            if (this.undoStack.length > 1) {
                undo.classList.remove('disabled')
            } else {
                undo.classList.add('disabled')
            }

            if (this.redoStack.length > 0) {
                redo.classList.remove('disabled')
            } else {
                redo.classList.add('disabled')
            }
        }

        addDataToUndoStackDebounceTimer = null

        addDataToUndoStack() {
            clearTimeout(this.addDataToUndoStackDebounceTimer);

            this.addDataToUndoStackDebounceTimer = setTimeout(() => this.addDataToUndoStackActual(), 50);
        }

        addDataToUndoStackActual() {
            const fetched = /** @type {{data: Record}} */
                this.fetch();

            const item = Espo.Utils.cloneDeep(fetched.data);

            this.undoStack.push(item);

            if (this.undoStack.length > this.undoStackLength) {
                this.undoStack.shift();
            }

            this.redoStack = [];

            this.controlUndoActions();
        }

        undo() {
            if (this.undoStack.length <= 1) {
                return;
            }

            /** @type {{list: Record[], createdEntitiesData: Record}} */
            const item = this.undoStack.pop();

            this.redoStack.push(item);

            const last = this.undoStack[this.undoStack.length - 1];

            const cloned = Espo.Utils.cloneDeep(last);

            this.dataList = cloned.list;
            this.createdEntitiesData = cloned.createdEntitiesData;

            this.fetchToModel();
            this.reRender();

            this.trigger('change');

            this.controlUndoActions();
        }

        redo() {
            if (this.redoStack.length === 0) {
                return;
            }

            const item = this.redoStack.pop();

            this.undoStack.push(item);

            const cloned = Espo.Utils.cloneDeep(item);

            this.dataList = cloned.list;
            this.createdEntitiesData = cloned.createdEntitiesData;

            this.fetchToModel();
            this.reRender();

            this.trigger('change');

            this.controlUndoActions();
        }
    }
});
