<?php
/*
 * This file is part of  the extension: Ebla Multiple Link Pro
 * Copyright (c) Eblasoft Bilişim Ltd.
 *
 * This Software is the property of Eblasoft Bilişim Ltd. and is protected
 * by copyright law - it is NOT Freeware and can be used only in one project
 * under a proprietary license, which is delivered along with this program.
 * If not, see <http://eblasoft.com.tr/eula>.
 *
 * This Software is distributed as is, with LIMITED WARRANTY AND LIABILITY.
 * Any unauthorised use of this Software without a valid license is
 * a violation of the License Agreement.
 *
 * According to the terms of the license you shall not resell, sublicense,
 * rent, lease, distribute or otherwise transfer rights or usage of this
 * Software or its derivatives. You may modify the code of this Software
 * for your own needs, if source code is provided.
 */

namespace Espo\Modules\EblaLinkPro\Core\FieldProcessing\LinkMultiple;

use Espo\ORM\Entity;

use Espo\Core\{
    FieldProcessing\Loader as LoaderInterface,
    FieldProcessing\Loader\Params,
    ORM\EntityManager,
    Record\ServiceContainer as RecordServiceContainer,
    Select\SearchParams,
    Utils\FieldUtil,
    Utils\Log,
    Utils\Metadata
};

use Espo\ORM\Defs as OrmDefs;
use Espo\Tools\LayoutManager\LayoutManager;
use Exception;

class ListLoader implements LoaderInterface
{
    private $scope;

    use inlineLinksList;

    public function __construct(
        private readonly OrmDefs $ormDefs,
        private readonly EntityManager $entityManager,
        private readonly RecordServiceContainer $recordServiceContainer,
        private readonly Metadata $metadata,
        private readonly LayoutManager $layoutManager,
        private readonly Log $log,
        private readonly FieldUtil $fieldUtil,
    ) {}

    public function process(Entity $entity, Params $params): void
    {
        foreach ($this->getFieldList($entity->getEntityType()) as $field) {
            $this->loadList($entity, $field);
        }
    }

    public function loadList(Entity $entity, string $field)
    {
        if (!$field) {
            return;
        }

        $defs = $this->ormDefs
            ->getEntity($entity->getEntityType())
            ->getField($field);

        $service = $this->recordServiceContainer->get($entity->getEntityType());
        $holderAttr = $field . 'ListHolder';

        $layout = $defs->getParam('layout');
        $linkEntityType = $this->ormDefs->getEntity($entity->getEntityType())->getRelation($field)->getParam('entity');

        $columns = [];

        if ($layout === 'custom') {
            $columns = $defs->getParam('fields');
        } else {
            $layout = $layout ?: 'list';
            $layoutData = json_decode($this->layoutManager->get($linkEntityType, $layout) ?? '[]');

            if (is_array($layoutData)) {
                $columns = array_map(function ($column) {
                    return $column->name;
                }, $layoutData);
            }
        }

        $editLayout = $defs->getParam('editLayout');

        if ($editLayout) {
            $editLayoutData = json_decode($this->layoutManager->get($linkEntityType, $editLayout) ?? '[]');
            if (is_array($editLayoutData)) {
                $columns = array_map(function ($column) {
                    return $column->name;
                }, $editLayoutData);
            }
        }

        $columns = array_unique($columns);

        try {
            $searchParams = [
                'orderBy' => $defs->getParam('orderBy'),
                'order' => $defs->getParam('order'),
            ];

            // id columns length > 1 then use id columns as select
            if (count($columns) > 1) {
                $searchParams['select'] = $this->getAttributeListFromColumns($linkEntityType, $columns);
            }

            $recordCollection = $service->findLinked(
                $entity->getId(),
                $field,
                SearchParams::fromRaw($searchParams)
            );

            $entity->set($holderAttr, $recordCollection->getValueMapList());

            if (!$entity->isNew() && !$entity->hasFetched($holderAttr)) {
                $entity->setFetched($holderAttr, $recordCollection->getValueMapList());
            }
        } catch (Exception $exception) {
            $this->log->error($exception->getMessage());
            $entity->set($holderAttr, []);
        }
    }

    private function getAttributeListFromColumns($scope, $columns): array
    {
        $list = [];
        foreach ($columns as $column) {
            $type = $this->ormDefs->getEntity($scope)->getField($column)->getType();
            if (!$type) {
                continue;
            }

            $attributeList = $this->fieldUtil->getAttributeList($scope, $column);
            foreach ($attributeList as $attribute) {
                $list[] = $attribute;
            }
        }

        return $list;
    }
}
