<?php
/*
 * This file is part of  the extension: Ebla Multiple Link Pro
 * Copyright (c) Eblasoft Bilişim Ltd.
 *
 * This Software is the property of Eblasoft Bilişim Ltd. and is protected
 * by copyright law - it is NOT Freeware and can be used only in one project
 * under a proprietary license, which is delivered along with this program.
 * If not, see <http://eblasoft.com.tr/eula>.
 *
 * This Software is distributed as is, with LIMITED WARRANTY AND LIABILITY.
 * Any unauthorised use of this Software without a valid license is
 * a violation of the License Agreement.
 *
 * According to the terms of the license you shall not resell, sublicense,
 * rent, lease, distribute or otherwise transfer rights or usage of this
 * Software or its derivatives. You may modify the code of this Software
 * for your own needs, if source code is provided.
 */

namespace Espo\Modules\EblaLinkPro\Controllers;

use Espo\Core\Acl;
use Espo\Core\Acl\Table as AclTable;
use Espo\Core\Api\Request;
use Espo\Core\Binding\BindingContainer;
use Espo\Core\Binding\BindingContainerBuilder;
use Espo\Core\Controllers\Record;
use Espo\Core\Di\MetadataAware;
use Espo\Core\Di\MetadataSetter;
use Espo\Core\Exceptions\BadRequest;
use Espo\Core\Exceptions\Error;
use Espo\Core\Exceptions\Forbidden;
use Espo\Core\Exceptions\NotFound;
use Espo\Core\Record\HookManager;
use Espo\Entities\User;

class EblaUnlink extends Record implements MetadataAware
{
    use MetadataSetter;

    private HookManager $recordHookManager;

    /**
     * Un relate records.
     */
    public function deleteActionRemoveLink(Request $request): bool
    {
        $id = $request->getRouteParam('id');
        $link = $request->getRouteParam('link');
        $scope = $request->getRouteParam('scope');

        $noEditAccessRequiredForUnLink = $this->metadata->get(['entityDefs', $scope, 'fields', $link, 'noEditAccessRequiredForUnLink']);

        $data = $request->getParsedBody();

        if (!$id || !$link) {
            throw new BadRequest();
        }

        $foreignIdList = [];

        if (isset($data->id)) {
            $foreignIdList[] = $data->id;
        }

        if (isset($data->ids) && is_array($data->ids)) {
            foreach ($data->ids as $foreignId) {
                $foreignIdList[] = $foreignId;
            }
        }

        $result = false;

        $service = $this->getRecordService($scope);

        foreach ($foreignIdList as $foreignId) {
            if ($noEditAccessRequiredForUnLink) {
                $this->unlink($scope, $id, $link, $foreignId);
            } else {
                $service->unlink($id, $link, $foreignId);
            }

            $result = true;
        }

        return $result;
    }

    public function unlink(string $scope, string $id, string $link, string $foreignId): void
    {
        $service = $this->getRecordService($scope);

        if (!$this->acl->check($scope)) {
            throw new Forbidden();
        }

        if (empty($scope) || empty($id) || empty($link) || empty($foreignId)) {
            throw new BadRequest;
        }

        $entity = $this->entityManager->getRepository($scope)->getById($id);

        if (!$entity) {
            throw new NotFound();
        }

        if (!$this->acl->check($entity, AclTable::ACTION_EDIT)) {
            throw new Forbidden();
        }

        $methodName = 'unlink' . ucfirst($link);

        if ($link !== 'entity' && method_exists($service, $methodName)) {
            $service->$methodName($id, $foreignId);

            return;
        }

        $foreignEntityType = $entity->getRelationParam($link, 'entity');

        if (!$foreignEntityType) {
            throw new Error("Entity '{$scope}' has not relation '{$link}'.");
        }

        $foreignEntity = $this->entityManager->getEntity($foreignEntityType, $foreignId);

        if (!$foreignEntity) {
            throw new NotFound();
        }

        $this->getRecordHookManager()->processBeforeUnlink($entity, $link, $foreignEntity);

        $this->entityManager->getRepository($scope)->unrelate($entity, $link, $foreignEntity);
    }

    private function createBinding(): BindingContainer
    {
        return BindingContainerBuilder::create()
            ->bindInstance(User::class, $this->user)
            ->bindInstance(Acl::class, $this->acl)
            ->build();
    }

    private function getRecordHookManager(): HookManager
    {
        if (!$this->recordHookManager) {
            $this->recordHookManager =
                $this->injectableFactory->createWithBinding(HookManager::class, $this->createBinding());
        }

        return $this->recordHookManager;
    }
}
