<?php

/**
 * This file is part of the eNUBE Custom - EspoCRM extension.
 *
 * dubas s.c. - contact@dubas.pro
 * Copyright (C) 2023-2025 Arkadiy Asuratov, Emil Dubielecki
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

namespace Espo\Modules\DubasCustom\Tools\DashboardSet;

use Error;
use Espo\Core\DataManager;
use Espo\Core\Select\SelectBuilderFactory;
use Espo\Core\Utils\Config;
use Espo\Core\Utils\Config\ConfigWriter;
use Espo\Core\Utils\File\Manager as FileManager;
use Espo\Core\Utils\Json;
use Espo\Core\Utils\Language;
use Espo\Core\Utils\Metadata;
use Espo\Entities\DashboardTemplate;
use Espo\Entities\User;
use Espo\Modules\DubasCustom\Entities\DashboardSet;
use Espo\ORM\EntityManager;
use Exception;

class Service
{
    public function __construct(
        private readonly EntityManager $entityManager,
        private readonly Config $config,
        private readonly ConfigWriter $configWriter,
        private readonly Metadata $metadata,
        private readonly FileManager $fileManager,
        private readonly DataManager $dataManager,
        private readonly Language $language,
        private readonly Language $baseLanguage,
        private readonly SelectBuilderFactory $selectBuilderFactory,
        private readonly User $user
    ) {}

    /**
     * @return string[]
     */
    public function getUserDashboardSetList(?string $userId = null): array
    {
        if (empty($userId)) {
            $userId = $this->user->getId();
        }

        $query = $this->selectBuilderFactory
            ->create()
            ->from(DashboardSet::ENTITY_TYPE)
            ->withStrictAccessControl()
            ->buildQueryBuilder()
            ->select([
                'number',
            ])
            ->build();

        /** @var DashboardSet[] $collection */
        $collection = $this->entityManager
            ->getRDBRepositoryByClass(DashboardSet::class)
            ->clone($query)
            ->find();

        $list = [];

        foreach ($collection as $entity) {
            $list[] = DashboardSet::ENTITY_TYPE . $entity->getNumber();
        }

        return $list;
    }

    public function saveDefs(?string $id): void
    {
        $entity = $this->getDashboardSet($id);
        $name = 'DashboardSet' . $entity->getNumber();

        $this->saveClientDefs($name, $entity);
        $this->saveScopeDefs($name);
        $this->saveAclDefs($name);

        $this->baseLanguage->set('Global', 'scopeNames', $name, $entity->getName());
        $this->baseLanguage->set('Global', 'scopeNamesPlural', $name, $entity->getName());

        $this->language->set('Global', 'scopeNames', $name, $entity->getName());
        $this->language->set('Global', 'scopeNamesPlural', $name, $entity->getName());

        $this->metadata->save();
        $this->configWriter->save();

        $this->baseLanguage->save();
        $this->language->save();

        $this->dataManager->clearCache();
    }

    public function removeDefs(?string $id): void
    {
        $entity = $this->getDashboardSet($id);
        $name = 'DashboardSet' . $entity->getNumber();

        $this->metadata->delete('clientDefs', $name);
        $this->metadata->delete('scopes', $name);

        $this->fileManager->removeFile("custom/Espo/Custom/Resources/metadata/clientDefs/{$name}.json");
        $this->fileManager->removeFile("custom/Espo/Custom/Resources/metadata/scopes/{$name}.json");

        try {
            $this->language->delete('Global', 'scopeNames', $name);
            $this->language->delete('Global', 'scopeNamesPlural', $name);

            $this->baseLanguage->delete('Global', 'scopeNames', $name);
            $this->baseLanguage->delete('Global', 'scopeNamesPlural', $name);
        } catch (Exception) {
        }

        $this->language->save();

        if ($this->isLanguageNotBase()) {
            $this->baseLanguage->save();
        }

        $this->dataManager->clearCache();
    }

    private function getDashboardSet(?string $id): DashboardSet
    {
        if (empty($id)) {
            throw new Error('Dashboard Set ID is required.');
        }

        /**
         * Reloads the entity to ensure it has the correct number.
         *
         * @var DashboardSet $entity The dashboard set entity to reload.
         */
        $entity = $this->entityManager->getEntityById(DashboardSet::ENTITY_TYPE, $id);

        if (!$entity) {
            throw new Error("Dashboard Set {$id} is not found.");
        }

        if (!$entity->getNumber()) {
            throw new Error("Dashboard Set must have a number.");
        }

        return $entity;
    }

    private function saveClientDefs(string $name, DashboardSet $entity): void
    {
        $dashboardTemplateId = $entity->get('dashboardTemplateId');

        if (!is_string($dashboardTemplateId)) {
            throw new Error('Dashboard Template ID is required.');
        }

        $dashboardTemplate = $this->entityManager
            ->getRDBRepositoryByClass(DashboardTemplate::class)
            ->getById($dashboardTemplateId);

        if (!$dashboardTemplate) {
            throw new Error('Dashboard Template not found.');
        }

        /** @var array<string, array{layout: array<int, array{id: string}>}> $layout */
        $layout = $dashboardTemplate->get('layout') ?? [];
        $layout = $this->updateLayoutIds($name, $layout);

        /** @var string $contents */
        $contents = $this->fileManager->getContents(
            'custom/Espo/Modules/DubasCustom/Templates/Metadata/clientDefs.json'
        );

        /** @var array<string,mixed> $defsData */
        $defsData = Json::decode($contents, true);

        $metadata = $this->metadata->get('clientDefs', $name) ?? [];

        if (is_array($metadata) && !array_key_exists('iconClass', $metadata)) {
            $defsData['iconClass'] = 'fas fa-th-large';
        }

        $defsData['dashboardLayout'] = $layout;

        $configDashletsOptions = $this->config->get('dashletsOptions');
        if (!is_object($configDashletsOptions)) {
            $configDashletsOptions = (object) [];
        }

        $dashletsOptions = $dashboardTemplate->get('dashletsOptions');
        if (!is_object($dashletsOptions)) {
            $dashletsOptions = (object) [];
        }

        foreach (get_object_vars($dashletsOptions) as $key => $options) {
            $key = "{$name}__{$key}";
            $configDashletsOptions->$key = $options;
        }

        $this->configWriter->set('dashletsOptions', $configDashletsOptions);
        $this->metadata->set('clientDefs', $name, $defsData);
    }

    private function saveScopeDefs(string $name): void
    {
        /** @var string $contents */
        $contents = $this->fileManager->getContents(
            'custom/Espo/Modules/DubasCustom/Templates/Metadata/scopes.json'
        );

        /** @var array<string,mixed> $defsData */
        $defsData = Json::decode($contents, true);

        $this->metadata->set('scopes', $name, $defsData);
    }

    private function saveAclDefs(string $name): void
    {
        /** @var string $contents */
        $contents = $this->fileManager->getContents(
            'custom/Espo/Modules/DubasCustom/Templates/Metadata/acl.json'
        );
        $contents = str_replace('{entityType}', $name, $contents);

        /** @var array<string,mixed> $defsData */
        $defsData = Json::decode($contents, true);

        $this->metadata->set('app', 'acl', $defsData);
    }

    /**
     * @param array<string, array{layout: array<int, array{id: string}>}> $data
     * @return array<string, array{layout: array<int, array{id: string}>}>
     */
    private function updateLayoutIds(string $name, array $data = []): array
    {
        /** @var array<string, array{layout: array<int, array{id: string}>}> $data */
        $data = Json::decode(Json::encode($data), true);

        foreach ($data as &$item) {
            if (isset($item['layout']) && is_array($item['layout'])) {
                foreach ($item['layout'] as &$layout) {
                    if (isset($layout['id']) && is_string($layout['id'])) {
                        $layout['id'] = $name . '__' . $layout['id'];
                    }
                }
            }
        }

        return $data;
    }

    private function isLanguageNotBase(): bool
    {
        return $this->language->getLanguage() !== $this->baseLanguage->getLanguage();
    }
}
