<?php
/***********************************************************************************
 * The contents of this file are subject to the Extension License Agreement
 * ("Agreement") which can be viewed at
 * https://www.espocrm.com/extension-license-agreement/.
 * By copying, installing downloading, or using this file, You have unconditionally
 * agreed to the terms and conditions of the Agreement, and You may not use this
 * file except in compliance with the Agreement. Under the terms of the Agreement,
 * You shall not license, sublicense, sell, resell, rent, lease, lend, distribute,
 * redistribute, market, publish, commercialize, or otherwise transfer rights or
 * usage to the software or any modified version or derivative work of the software
 * created by or for you.
 *
 * Copyright (C) 2024-2025 Letrium Ltd.
 *
 * License ID: f27e70ce6801a13265271f5669c8bc5c
 ************************************************************************************/

namespace Espo\Modules\Project\Tools\ProjectTask\Api;

use Espo\Core\Acl;
use Espo\Core\Api\Action;
use Espo\Core\Api\Request;
use Espo\Core\Api\Response;
use Espo\Core\Api\ResponseComposer;
use Espo\Core\Exceptions\BadRequest;
use Espo\Core\Exceptions\Forbidden;
use Espo\Core\Exceptions\NotFound;
use Espo\Core\Record\EntityProvider;
use Espo\Modules\Project\Entities\Project;
use Espo\Modules\Project\Entities\ProjectTask;
use Espo\Modules\Project\Tools\ProjectTask\RecordService;

class PostMoveInGroup implements Action
{
    public function __construct(
        private EntityProvider $entityProvider,
        private Acl $acl,
        private RecordService $recordService,
    ) {}

    public function process(Request $request): Response
    {
        $id = $request->getRouteParam('id') ?? throw new BadRequest();
        $placement = $request->getParsedBody()->placement ?? null;

        if (!in_array($placement, [RecordService::PLACEMENT_TOP, RecordService::PLACEMENT_BOTTOM])) {
            throw new BadRequest("Bad 'placement'.");
        }

        $task = $this->getTask($id);

        $this->recordService->moveInGroup($task, $placement);

        return ResponseComposer::json(true);
    }

    /**
     * @throws Forbidden
     * @throws NotFound
     */
    private function getTask(string $id): ProjectTask
    {
        $task = $this->entityProvider->getByClass(ProjectTask::class, $id);

        if (!$this->acl->checkEntityEdit($task)) {
            throw new Forbidden("No 'edit' access.");
        }

        if (!$task->getProjectId()) {
            throw new Forbidden("No 'project'.");
        }

        $project = $this->entityProvider->getByClass(Project::class, $task->getProjectId());

        if (!$this->acl->checkEntityEdit($project)) {
            throw new Forbidden("No 'edit' access to project.");
        }

        return $task;
    }
}
