<?php
/***********************************************************************************
 * The contents of this file are subject to the Extension License Agreement
 * ("Agreement") which can be viewed at
 * https://www.espocrm.com/extension-license-agreement/.
 * By copying, installing downloading, or using this file, You have unconditionally
 * agreed to the terms and conditions of the Agreement, and You may not use this
 * file except in compliance with the Agreement. Under the terms of the Agreement,
 * You shall not license, sublicense, sell, resell, rent, lease, lend, distribute,
 * redistribute, market, publish, commercialize, or otherwise transfer rights or
 * usage to the software or any modified version or derivative work of the software
 * created by or for you.
 *
 * Copyright (C) 2024-2025 Letrium Ltd.
 *
 * License ID: f27e70ce6801a13265271f5669c8bc5c
 ************************************************************************************/

namespace Espo\Modules\Project\Classes\MassAction\ProjectTask;

use Espo\Core\Acl;
use Espo\Core\Acl\Table;
use Espo\Core\Exceptions\BadRequest;
use Espo\Core\Exceptions\Forbidden;
use Espo\Core\MassAction\Data;
use Espo\Core\MassAction\MassAction;
use Espo\Core\MassAction\Params;
use Espo\Core\MassAction\QueryBuilder;
use Espo\Core\MassAction\Result;
use Espo\Entities\User;
use Espo\Modules\Project\Entities\ProjectTask;
use Espo\Modules\Project\Tools\Project\MemberTaskAccessChecker;
use Espo\Modules\Project\Tools\ProjectTask\RecordService;
use Espo\ORM\EntityManager;

class Complete implements MassAction
{
    public function __construct(
        private Acl $acl,
        private QueryBuilder $queryBuilder,
        private EntityManager $entityManager,
        private RecordService $recordService,
        private User $user,
        private MemberTaskAccessChecker $memberTaskAccessChecker,
    ) {}

    public function process(Params $params, Data $data): Result
    {
        if (!$this->acl->checkScope(ProjectTask::ENTITY_TYPE, Table::ACTION_EDIT)) {
            throw new Forbidden("No 'edit' access.");
        }

        if (!$params->hasIds()) {
            throw new BadRequest("Mass-update with search params is not implemented.");
        }

        $query = $this->queryBuilder->build($params);

        /** @var iterable<ProjectTask> $collection */
        $collection = $this->entityManager
            ->getRDBRepositoryByClass(ProjectTask::class)
            ->clone($query)
            ->find();

        $count = 0;

        foreach ($collection as $entity) {
            $done = $this->processEntity($entity);

            if ($done) {
                $count++;
            }
        }

        return new Result($count);
    }

    private function processEntity(ProjectTask $entity): bool
    {
        if (!$this->checkAccess($entity)) {
            return false;
        }

        try {
            $this->recordService->complete($entity);
        } catch (Forbidden) {
            return false;
        }

        return true;
    }

    private function checkAccess(ProjectTask $entity): bool
    {
        if ($this->acl->checkEntityEdit($entity)) {
            return true;
        }

        return $this->memberTaskAccessChecker->checkColumnEdit($this->user, $entity);
    }
}
