<?php
/***********************************************************************************
 * The contents of this file are subject to the Extension License Agreement
 * ("Agreement") which can be viewed at
 * https://www.espocrm.com/extension-license-agreement/.
 * By copying, installing downloading, or using this file, You have unconditionally
 * agreed to the terms and conditions of the Agreement, and You may not use this
 * file except in compliance with the Agreement. Under the terms of the Agreement,
 * You shall not license, sublicense, sell, resell, rent, lease, lend, distribute,
 * redistribute, market, publish, commercialize, or otherwise transfer rights or
 * usage to the software or any modified version or derivative work of the software
 * created by or for you.
 *
 * Copyright (C) 2024-2025 Letrium Ltd.
 *
 * License ID: f27e70ce6801a13265271f5669c8bc5c
 ************************************************************************************/

namespace Espo\Modules\Project\Tools\Project\Api;

use Espo\Core\Acl;
use Espo\Core\Api\Action;
use Espo\Core\Api\Request;
use Espo\Core\Api\Response;
use Espo\Core\Api\ResponseComposer;
use Espo\Core\Exceptions\BadRequest;
use Espo\Core\Exceptions\Forbidden;
use Espo\Core\Exceptions\NotFound;
use Espo\Core\Field\Date;
use Espo\Core\Record\EntityProvider;
use Espo\Modules\Project\Entities\Project;
use Espo\Modules\Project\Tools\Project\Cloning\CloneService;
use Espo\Modules\Project\Tools\Project\Cloning\Params;

/**
 * @noinspection PhpUnused
 */
class PostClone implements Action
{
    public function __construct(
        private EntityProvider $entityProvider,
        private Acl $acl,
        private CloneService $service,
    ) {}


    public function process(Request $request): Response
    {
        $id = $request->getRouteParam('id') ?? throw new BadRequest();

        $params = $this->fetchParams($request);
        $project = $this->getProject($id);

        $result = $this->service->clone($project, $params);

        return ResponseComposer::json([
            'id' => $result->id,
        ]);
    }

    /**
     * @throws Forbidden
     * @throws NotFound
     */
    private function getProject(string $id): Project
    {
        $project = $this->entityProvider->getByClass(Project::class, $id);

        if (!$this->acl->checkScope(Project::ENTITY_TYPE, Acl\Table::ACTION_CREATE)) {
            throw new Forbidden("No 'create' access to be able to clone.");
        }

        if (!$this->acl->checkEntityEdit($project)) {
            throw new Forbidden("No 'edit' access to record to be able to clone.");
        }

        return $project;
    }

    /**
     * @throws BadRequest
     */
    private function fetchParams(Request $request): Params
    {
        $data = $request->getParsedBody();

        if (!isset($data->name) && !is_string($data->name) || $data->name === '') {
            throw new BadRequest("No name.");
        }

        $dateStartString = $data->dateStart ?? null;

        if ($dateStartString && !is_string($dateStartString)) {
            throw new BadRequest("Bad dateStart.");
        }

        $dateStart = $dateStartString ? Date::fromString($dateStartString) : null;
        $withMembers = (bool) ($data->withMembers ?? false);
        $withTasks = (bool) ($data->withTasks ?? false);
        $withAssignees = ($data->withAssignees ?? false) && $withMembers;
        $withOwners = ($data->withOwners ?? false) && $withMembers;
        $withAttachments = (bool) ($data->withAttachments ?? false);

        if (!$dateStart && $withTasks) {
            throw new BadRequest("No dateStart.");
        }

        return new Params(
            name: $data->name,
            dateStart: $dateStart,
            withMembers: $withMembers,
            withTasks: $withTasks,
            withOwners: $withOwners,
            withAssignees: $withAssignees,
            withAttachments: $withAttachments,
        );
    }
}
