<?php
/***********************************************************************************
 * The contents of this file are subject to the Extension License Agreement
 * ("Agreement") which can be viewed at
 * https://www.espocrm.com/extension-license-agreement/.
 * By copying, installing downloading, or using this file, You have unconditionally
 * agreed to the terms and conditions of the Agreement, and You may not use this
 * file except in compliance with the Agreement. Under the terms of the Agreement,
 * You shall not license, sublicense, sell, resell, rent, lease, lend, distribute,
 * redistribute, market, publish, commercialize, or otherwise transfer rights or
 * usage to the software or any modified version or derivative work of the software
 * created by or for you.
 *
 * Copyright (C) 2024-2025 Letrium Ltd.
 *
 * License ID: f27e70ce6801a13265271f5669c8bc5c
 ************************************************************************************/

namespace Espo\Modules\Project\Hooks\ProjectTask;

use Espo\Core\Hook\Hook\BeforeSave;
use Espo\Modules\Project\Entities\ProjectGroup;
use Espo\Modules\Project\Entities\ProjectTask;
use Espo\ORM\Entity;
use Espo\ORM\EntityManager;
use Espo\ORM\Query\Part\Order;
use Espo\ORM\Repository\Option\SaveOptions;

/**
 * @implements BeforeSave<ProjectTask>
 */
class Prepare implements BeforeSave
{
    public const OPTION_SKIP_ORDER = 'skipProjectOrder';

    public function __construct(
        private EntityManager $entityManager,
    ) {}

    public function beforeSave(Entity $entity, SaveOptions $options): void
    {
        if ($options->get(self::OPTION_SKIP_ORDER)) {
            return;
        }

        if ($entity->isNew() || $entity->isAttributeChanged('groupId')) {
            $this->processOrder($entity);
        }

        if ($entity->isNew() || $entity->isAttributeChanged('groupId') || $entity->isAttributeChanged('columnId')) {
            $this->processBoardOrder($entity);
        }
    }

    private function processOrder(ProjectTask $entity): void
    {
        $group = $this->getGroup($entity);

        $top = $group->getTaskPlacement() === ProjectGroup::TASK_PLACEMENT_TOP;

        $one = $this->entityManager
            ->getRDBRepositoryByClass(ProjectTask::class)
            ->select('order')
            ->where([
                'groupId' => $entity->getGroupId(),
                'parentTaskId' => $entity->getParentTaskId(),
            ])
            ->order('order', $top ? Order::ASC : Order::DESC)
            ->findOne();

        $order = 0;

        if ($one) {
            $order = $top ?
                $one->getOrder() - 10 :
                $one->getOrder() + 10;
        }

        $entity->setOrder($order);
    }

    private function getGroup(ProjectTask $entity): ?ProjectGroup
    {
        if (!$entity->getGroupId()) {
            return null;
        }

        return $this->entityManager->getRDBRepositoryByClass(ProjectGroup::class)->getById($entity->getGroupId());
    }

    private function processBoardOrder(ProjectTask $entity): void
    {
        if ($entity->getParentTaskId()) {
            return;
        }

        $one = $this->entityManager
            ->getRDBRepositoryByClass(ProjectTask::class)
            ->select('boardOrder')
            ->where([
                'groupId' => $entity->getGroupId(),
                'parentTaskId' => null,
                'columnId' => $entity->getColumnId(),
            ])
            ->order('boardOrder', Order::ASC)
            ->findOne();

        $order = 0;

        if ($one) {
            $order = $one->getBoardOrder() - 10;
        }

        $entity->setBoardOrder($order);
    }
}
