<?php
/*
 * This file is part of  the extension: Ebla Multiple Link Pro
 * Copyright (c) Eblasoft Bilişim Ltd.
 *
 * This Software is the property of Eblasoft Bilişim Ltd. and is protected
 * by copyright law - it is NOT Freeware and can be used only in one project
 * under a proprietary license, which is delivered along with this program.
 * If not, see <http://eblasoft.com.tr/eula>.
 *
 * This Software is distributed as is, with LIMITED WARRANTY AND LIABILITY.
 * Any unauthorised use of this Software without a valid license is
 * a violation of the License Agreement.
 *
 * According to the terms of the license you shall not resell, sublicense,
 * rent, lease, distribute or otherwise transfer rights or usage of this
 * Software or its derivatives. You may modify the code of this Software
 * for your own needs, if source code is provided.
 */

namespace Espo\Modules\EblaLinkPro\Core\FieldProcessing\Relation;

use Espo\Core\Acl;
use Espo\Core\Acl\Table as AclTable;
use Espo\Core\Binding\BindingContainer;
use Espo\Core\Binding\BindingContainerBuilder;
use Espo\Core\Exceptions\BadRequest;
use Espo\Core\Exceptions\Error;
use Espo\Core\Exceptions\Forbidden;
use Espo\Core\Exceptions\NotFound;
use Espo\Core\InjectableFactory;
use Espo\Core\FieldProcessing\Saver\Params;
use Espo\Core\FieldValidation\Exceptions\ValidationError;
use Espo\Core\ORM\EntityManager;
use Espo\Core\Record\CreateParams;
use Espo\Core\Record\HookManager as RecordHookManager;
use Espo\Core\Record\ServiceContainer as RecordServiceContainer;
use Espo\Core\Record\UpdateParams;
use Espo\Core\Utils\FieldUtil;
use Espo\Core\Utils\Log;
use Espo\Core\Utils\Metadata;
use Espo\Modules\EblaLinkPro\Core\FieldProcessing\LinkMultiple\inlineLinksList;
use Espo\Modules\EblaLinkPro\Core\FieldProcessing\LinkMultiple\ListLoader;
use Espo\Core\FieldProcessing\Loader\Params as LoaderParams;
use Espo\ORM\Defs as OrmDefs;
use Espo\Core\ORM\Entity as CoreEntity;
use Espo\ORM\Type\AttributeType;
use Exception;

class LinkMultipleSaver extends \Espo\Core\FieldProcessing\Relation\LinkMultipleSaver
{
    public const JSON_ARRAY = AttributeType::JSON_ARRAY;
    public const JSON_OBJECT = AttributeType::JSON_OBJECT;

    use inlineLinksList;

    private RecordServiceContainer $recordServiceContainer;
    private Acl $acl;

    public function __construct(
        private OrmDefs $ormDefs,
        private EntityManager $entityManager,
        private Log $log,
        private Metadata $metadata,
        private FieldUtil $fieldUtil,
        private InjectableFactory $injectableFactory,
        private ListLoader $listLoader,
    ) {
        parent::__construct($entityManager);
    }

    public function process(CoreEntity $entity, string $name, Params $params): void
    {
        $entityType = $entity->getEntityType();
        $entityDefs = $this->ormDefs->getEntity($entityType);

        if (!$entityDefs->hasField($name)) {
            parent::process($entity, $name, $params);
            return;
        }

        $enableForm = $entityDefs->getField($name)->getParam('enableForm');
        $list = $entity->get($name . 'ListHolder');

        $relationDefs = $entityDefs->getRelation($name);

        // if many to many, we only support view mode.
        if ($relationDefs->hasRelationshipName()) {
            parent::process($entity, $name, $params);
            $this->listLoader->loadList($entity, $name);
            return;
        }

        if (!$enableForm || empty($list) || !is_array($list)) {
            parent::process($entity, $name, $params);
            return;
        }

        $this->recordServiceContainer = $this->injectableFactory->create(RecordServiceContainer::class);
        $this->acl = $this->injectableFactory->create(Acl::class);

        $childScope = $relationDefs->getParam('entity');
        $linkFieldName = $relationDefs->getParam('foreign');
        $childFields = $this->fieldUtil->getEntityTypeFieldList($childScope);

        if (!$linkFieldName || !$childScope) {
            parent::process($entity, $name, $params);
            return;
        }

        $childDefs = $this->ormDefs->getEntity($childScope);

        $nameRequired = (
            $childDefs->hasField('name') &&
            $childDefs->getField('name')->getParam('required')
        );

        $assignedUserRequired = (
            $childDefs->hasField('assignedUser') &&
            $childDefs->getField('assignedUser')->getParam('required')
        );

        $childService = $this->recordServiceContainer->get($childScope);

        $holderData = [];
        foreach ($list as $attrs) {
            $attrId = $attrs->id ?? null;

            $childEntity = null;

            unset($attrs->id);
            unset($attrs->modifiedAt);

            if (!empty($attrId)) {
                $childEntity = $this->entityManager->getEntity($childScope, $attrId);
            }

            $attrs->{$linkFieldName . 'Id'} = $entity->getId();

            if ($assignedUserRequired && empty(@$attrs->assignedUserId)) {
                $attrs->assignedUserId = $entity->get('assignedUserId') ?? null;
            }
            if ($nameRequired && empty(@$attrs->name)) {
                $attrs->name = $entity->get('name');
            }

            try {
                if (!$childEntity) {
                    $childEntity = $childService->create($attrs, CreateParams::create()->withSkipDuplicateCheck(false));
                } else {
                    $relationType = $childEntity->getRelationType($linkFieldName);
                    $hasEditAccess = $this->acl->check($childEntity, AclTable::ACTION_EDIT);

                    if ($relationType !== 'manyMany' && $relationType !== 'hasMany' && $hasEditAccess) { // belongsTo
                        $childEntity->set($attrs);

                        $entityChanged = false;
                        foreach ($childFields as $childField) {
                            $actualFields = $this->fieldUtil->getActualAttributeList($childScope, $childField);
                            foreach ($actualFields as $actualField) {
                                if ($childEntity->isAttributeChanged($actualField)) {
                                    $entityChanged = true;
                                } else {
                                    unset($attrs->{$actualField});
                                }
                            }
                        }

                        if ($entityChanged) {
                            $childEntity = $childService->update($attrId, $attrs, UpdateParams::create());
                        }
                    } else {
                        $childEntity = $this->entityManager->getEntityById($childScope, $attrId);
                        if (!$childEntity) {
                            throw new NotFound("Linked Entity '{$childScope}' with id '{$attrId}' not found.");
                        }

                        // todo: use relate function
                        $childEntity->set($linkFieldName . 'Id', $entity->getId());
                        $this->entityManager->saveEntity($childEntity, ['skipAll' => true]);
                    }
                }

                $holderData[] = $childEntity->getValueMap();
            } catch (Exception $exception) {
                // $noEditAccessRequiredForUnLink = $this->metadata->get(['entityDefs', $entityType, 'fields', $name, 'noEditAccessRequiredForUnLink']) ?? true;

                // has no acl edit access.
                if ($childEntity /*&& $noEditAccessRequiredForUnLink*/) {
                    $this->link($entityType, $entity->getId(), $name, $childEntity->getId());
                    $holderData[] = $childEntity->getValueMap();
                } else {
                    $this->log->error('EblaLinkPro Save Error: ' . $exception->getMessage());

                    if ($entity->isNew()) {
                        $this->entityManager->removeEntity($entity);
                    }

                    throw $exception;
                }

                if ($exception instanceof ValidationError) {
                    throw $exception;
                }
            }
        }

        $entity->set($name . 'ListHolder', $holderData);
        // update entity if any formula triggered by the linked entities.
        $updatedEntity = $this->entityManager->getEntityById($entityType, $entity->getId());
        if (!empty($updatedEntity)) {
            $entity->set($updatedEntity->getValueMap());
        }
    }

    /**
     * @throws BadRequest
     * @throws Forbidden
     * @throws Error
     * @throws NotFound
     */
    protected function link(string $scope, string $id, string $link, string $foreignId): void
    {
        if (!$this->acl->check($scope)) {
            throw new Forbidden();
        }

        if (empty($scope) || empty($id) || empty($link) || empty($foreignId)) {
            throw new BadRequest;
        }

        $repository = $this->entityManager->getRDBRepository($scope);
        $entity = $repository->getById($id);

        if (!$entity) {
            throw new NotFound();
        }

        if (!$this->acl->check($entity, AclTable::ACTION_EDIT)) {
            throw new Forbidden();
        }

        $service = $this->recordServiceContainer->get($scope);

        $methodName = 'link' . ucfirst($link);

        if ($link !== 'entity' && $link !== 'entityMass' && method_exists($service, $methodName)) {
            $service->$methodName($id, $foreignId);
            return;
        }

        $foreignEntityType = $entity->getRelationParam($link, 'entity');

        if (!$foreignEntityType) {
            throw new Error("Entity '{$scope}' has not relation '{$link}'.");
        }

        $foreignEntity = $this->entityManager->getEntity($foreignEntityType, $foreignId);

        if (!$foreignEntity) {
            throw new NotFound();
        }

        if (!$this->acl->check($foreignEntity, AclTable::ACTION_READ)) {
            throw new Forbidden();
        }

        $this->getRecordHookManager()->processBeforeLink($entity, $link, $foreignEntity);

        $repository->getRelation($entity, $link)->relate($foreignEntity); // need test
        // $repository->relate($entity, $link, $foreignEntity); // deprecated
    }

    private function createBinding(): BindingContainer
    {
        return BindingContainerBuilder::create()
            //->bindInstance(User::class, $this->user)
            ->bindInstance(Acl::class, $this->acl)
            ->build();
    }

    private function getRecordHookManager(): RecordHookManager
    {
        if (!$this->recordHookManager) {
            $this->recordHookManager =
                $this->injectableFactory->createWithBinding(RecordHookManager::class, $this->createBinding());
        }

        return $this->recordHookManager;
    }
}
