<?php

namespace Espo\Modules\EblaKanbanMultiple\Tools\Kanban;

use Espo\Core\{
    Acl\Table,
    AclManager,
    Exceptions\Error,
    Exceptions\ForbiddenSilent,
    InjectableFactory,
    Select\SearchParams,
    Utils\Config,
    Utils\Metadata};
use Espo\Entities\User;
use Espo\Tools\Kanban\Kanban;
use Espo\Tools\Kanban\Orderer;
use Espo\Tools\Kanban\Result;

class KanbanService extends \Espo\Tools\Kanban\KanbanService
{

    private $user;

    private $aclManager;

    private $injectableFactory;

    private $config;

    private $metadata;

    public function __construct(User $user, AclManager $aclManager, InjectableFactory $injectableFactory, Config $config, Metadata $metadata, Orderer $orderer)
    {
        parent::__construct($user, $aclManager, $injectableFactory, $config, $metadata, $orderer);
        $this->user = $user;
        $this->aclManager = $aclManager;
        $this->injectableFactory = $injectableFactory;
        $this->config = $config;
        $this->metadata = $metadata;
    }

    /**
     * @throws ForbiddenSilent
     * @throws Error
     */
    public function getData(string $entityType, SearchParams $searchParams): Result
    {
        $this->processAccessCheck($entityType);

        $disableCount = $this->metadata->get(['entityDefs', $entityType, 'collection', 'countDisabled']) ?? false;

        $orderDisabled = $this->metadata->get(['scopes', $entityType, 'kanbanOrderDisabled']) ?? false;

        $maxOrderNumber = $this->config->get('kanbanMaxOrderNumber');

        return $this->createKanban()
            ->setEntityType($entityType)
            ->setSearchParams($searchParams)
            ->setCountDisabled($disableCount)
            ->setOrderDisabled($orderDisabled)
            ->setUserId($this->user->getId())
            ->setMaxOrderNumber($maxOrderNumber)
            ->getResult();
    }

    private function createKanban(): Kanban
    {
        return $this->injectableFactory->create(EblaKanban::class);
    }

    private function processAccessCheck(string $entityType): void
    {
        if (!$this->metadata->get(['scopes', $entityType, 'object'])) {
            throw new ForbiddenSilent("Non-object entities are not supported.");
        }

        if ($this->metadata->get(['recordDefs', $entityType, 'kanbanDisabled'])) {
            throw new ForbiddenSilent("Kanban is disabled for '{$entityType}'.");
        }

        if (!$this->aclManager->check($this->user, $entityType, Table::ACTION_READ)) {
            throw new ForbiddenSilent();
        }
    }
}
